<?php
namespace TotalTheme\Topbar;

defined( 'ABSPATH' ) || exit;

/**
 * Topbar Core.
 *
 * @package TotalTheme
 * @subpackage Topbar
 * @version 5.7.0
 */
class Core {

	/**
	 * Topbar is enabled or not.
	 */
	protected static $is_enabled;

	/**
	 * Topbar has content or not.
	 */
	protected static $has_content;

	/**
	 * The topbar style.
	 */
	protected static $style;

	/**
	 * The topbar breakpoint.
	 */
	protected static $breakpoint;

	/**
	 * Stores the topbar template id if defined.
	 */
	protected static $template_id;

	/**
	 * Checks if the topbar is enabled or not.
	 */
	public static function is_enabled() {
		if ( ! is_null( self::$is_enabled ) ) {
			return self::$is_enabled;
		}

		$post_id = wpex_get_current_post_id();

		if ( wpex_elementor_location_exists( 'topbar' ) ) {
			$check = true;
		} else {
			$check = get_theme_mod( 'top_bar', true );
		}

		// Check meta.
		if ( $post_id && $meta = get_post_meta( $post_id, 'wpex_disable_top_bar', true ) ) {

			// Return false if disabled via post meta.
			if ( 'on' === $meta ) {
				$check = false;
			}

			// Return true if enabled via post meta.
			elseif ( 'enable' === $meta ) {
				$check = true;
			}

		}

		/**
		 * Filters whether the topbar is enabled or not.
		 *
		 * @param boolean $check
		 */
		$check = (bool) apply_filters( 'totaltheme/topbar/is_enabled', $check );

		/*** deprecated ***/
		$check = (bool) apply_filters( 'wpex_is_top_bar_enabled', $check ); // @deprecated
		self::$is_enabled = (bool) apply_filters( 'wpex_has_topbar', $check );

		return self::$is_enabled;
	}

	/**
	 * Returns the topbar style.
	 */
	public static function style() {
		if ( ! is_null( self::$style ) ) {
			return self::$style;
		}

		$style = get_theme_mod( 'top_bar_style' ) ?: 'one';

		/**
		 * Filters the topbar style.
		 *
		 * @param string $style
		 */
		self::$style = (string) apply_filters( 'wpex_top_bar_style', $style );

		return self::$style;
	}

	/**
	 * Checks if the topbar is set to fullwidth.
	 */
	public static function is_fullwidth() {
		if ( 'full-width' === wpex_site_layout() && get_theme_mod( 'top_bar_fullwidth' ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Checks if the topbar is sticky.
	 */
	public static function is_sticky() {
		if ( wpex_vc_is_inline() ) {
			return false;
		}

		$is_sticky = get_theme_mod( 'top_bar_sticky' );

		/**
		 * Filters whether the topbar is sticky or not.
		 *
		 * @param string $style
		 */
		$is_sticky = (bool) apply_filters( 'wpex_has_sticky_topbar', $is_sticky );

		return $is_sticky;
	}

	/**
	 * Returns the topbar breakpoint.
	 */
	public static function breakpoint() {
		if ( ! is_null( self::$breakpoint ) ) {
			return self::$breakpoint;
		}

		$breakpoint = get_theme_mod( 'topbar_split_breakpoint' );

		if ( 'none' !== $breakpoint && ( empty( $breakpoint ) || ! array_key_exists( $breakpoint, wpex_utl_breakpoints() ) ) ) {
			$breakpoint = 'md';
		}

		/**
		 * Filters the topbar breakpoint.
		 *
		 * @param string $breakpoint.
		 */
		$breakpoint = (string) apply_filters( 'totaltheme/topbar/breakpoint', $breakpoint );

		/*** deprecated ***/
		$breakpoint = (string) apply_filters( 'wpex_topbar_split_breakpoint', $breakpoint );

		self::$breakpoint = $breakpoint;

		return self::$breakpoint;
	}

	/**
	 * Return default content.
	 */
	public static function get_default_content() {
		$space = is_customize_preview() ? "\r\n\r\n" : '';
		return '[topbar_item icon="phone" text="1-800-987-654" link="tel:1-800-987-654"/]' . $space . '[topbar_item icon="envelope" text="admin@totalwptheme.com" link="mailto:admin@totalwptheme.com"/]' . $space . '[topbar_item type="login" icon="user" icon_logged_in="sign-out" text="User Login" text_logged_in="Log Out" logout_text="Logout"/]';
	}

	/**
	 * Return the topbar content from the theme mod.
	 */
	private static function get_content_from_mod() {
		return wpex_get_translated_theme_mod( 'top_bar_content', self::get_default_content() );
	}

	/**
	 * Return template ID.
	 */
	public static function get_template_id() {
		if ( ! is_null( self::$template_id ) ) {
			return self::$template_id;
		}
		$template_id = ''; // make sure it isn't null to prevent extra checks.
		$content = self::get_content_from_mod();
		if ( is_numeric( $content ) ) {
			$post_id = wpex_parse_obj_id( $content, 'page' );
			$post    = get_post( $post_id );
			if ( $post && ! is_wp_error( $post ) ) {
				$template_id = $post_id;
			}
		}
		self::$template_id = $template_id;
		return self::$template_id;
	}

	/**
	 * Return topbar content.
	 */
	public static function get_content() {
		$template_id = self::get_template_id();

		if ( $template_id ) {
			$content = wpex_parse_vc_content( get_post_field( 'post_content', $template_id ) );
		} else {
			$content = self::get_content_from_mod();
		}

		/**
		 * Filters the topbar content.
		 *
		 * @param string $content
		 */
		$content = (string) apply_filters( 'totaltheme/topbar/content', $content );

		/*** deprecated ***/
		$content = (string) apply_filters( 'wpex_top_bar_content', $content ); // @deprecated 5.4.5
		$content = (string) apply_filters( 'wpex_topbar_content', $content );

		return $content;
	}

	/**
	 * Check if the topbar has content.
	 */
	public static function has_content() {
		if ( ! is_null( self::$has_content ) ) {
			return self::$has_content;
		}

		if ( has_nav_menu( 'topbar_menu' ) || self::get_content() ) {
			self::$has_content = true;
		} else {
			self::$has_content = false;
		}

		return self::$has_content;
	}

	/**
	 * Returns wrapper classes.
	 */
	public static function get_wrapper_classes() {
		$classes = array();
		$style   = self::style();

		if ( self::is_fullwidth() ) {
			$classes[] = 'top-bar-full-width';
		}

		if ( self::is_sticky() ) {
			$classes[] = 'wpex-top-bar-sticky';
			$classes[] = 'wpex-z-99'; // @todo remove? we add another z-index when it's sticky is this still needed?
			$classes[] = 'wpex-surface-1';
		}

		if ( get_theme_mod( 'top_bar_bottom_border', true ) ) {
			$classes[] = 'wpex-border-b';
			$classes[] = 'wpex-border-main';
			$classes[] = 'wpex-border-solid';
		}

		$classes[] = 'wpex-text-sm';

		if ( $visibility = get_theme_mod( 'top_bar_visibility' ) ) {
			$classes[] = $visibility;
		}

		if ( 'three' === $style ) {
			$classes[] = 'textcenter';
		}

		/**
		 * Filters the topbar wrap class.
		 *
		 * @param array $class
		 */
		$classes = (array) apply_filters( 'totaltheme/topbar/wrapper_class', $classes );

		/*** deprecated ***/
		$classes = (array) apply_filters( 'wpex_topbar_wrap_class', $classes );
		$classes = apply_filters( 'wpex_get_topbar_classes', $classes );

		return implode( ' ', $classes );
	}

	/**
	 * Echo class attribute for the the topbar wrapper element.
	 */
	public static function wrapper_class() {
		$classes = self::get_wrapper_classes();
		if ( $classes ) {
			echo 'class="' . esc_attr( $classes ) . '"';
		}
	}

	/**
	 * Echo class attribute for the the topbar inner element.
	 */
	public static function inner_class() {
		$topbar_style = self::style();
		$split_bk     = self::breakpoint();
		$alignment    = wpex_get_mod( 'topbar_alignment', 'center', true );
		$has_split_bk = ( 'none' !== $split_bk );

		$classes = array(
			'container'
		);

		$classes[] = 'wpex-relative';  // !! important !!!
		$classes[] = 'wpex-py-15';

		if ( in_array( $topbar_style, array( 'one', 'two' ) ) ) {
			if ( $has_split_bk ) {
				$classes[] = 'wpex-' . sanitize_html_class( $split_bk ) . '-flex';
			} else {
				$classes[] = 'wpex-flex';
				$classes[] = 'wpex-overflow-x-auto';
				$classes[] = 'wpex-hide-scrollbar';
			}
			$classes[] = 'wpex-justify-between';
			$classes[] = 'wpex-items-center';
			if ( $has_split_bk ) {
				$classes[] = 'wpex-text-' . sanitize_html_class( $alignment );
				$classes[] = 'wpex-' . sanitize_html_class( $split_bk ) . '-text-initial';
			}
		}

		if ( 'one' === $topbar_style && ! self::has_content() ) {
			$classes[] = 'wpex-flex-row-reverse';
		}

		if ( 'two' === $topbar_style && ! Social::is_enabled() ) {
			$classes[] = 'wpex-flex-row-reverse';
		}

		if ( 'three' === $topbar_style ) {
			$classes[] = 'wpex-text-' . sanitize_html_class( $alignment );
		}

		if ( ! $has_split_bk ) {
			$classes[] = 'wpex-gap-30';
		}

		/**
		 * Filters the topbar inner class.
		 *
		 * @param array $class
		 */
		$classes = (array) apply_filters( 'totaltheme/topbar/inner_class', $classes );

		/*** deprecated ***/
		$classes = (array) apply_filters( 'wpex_topbar_class', $classes );

		if ( $classes ) {
			echo 'class="' . esc_attr( implode( ' ', $classes ) ) . '"';
		}
	}

	/**
	 * Returns content classes.
	 */
	public static function get_content_classes() {
		$classes      = array();
		$topbar_style = self::style();
		$split_bk     = self::breakpoint();
		$has_split_bk = ( 'none' !== $split_bk );

		if ( self::get_content() ) {
			$classes[] = 'has-content';
		}

		switch ( $topbar_style ) {
			case 'one':
				$classes[] = 'top-bar-left';
				break;
			case 'two':
				$classes[] = 'top-bar-right';
				if ( Social::is_enabled() && $has_split_bk ) {
					$classes[] = 'wpex-mt-10';
					$classes[] = 'wpex-' . $split_bk . '-mt-0';
				}
				break;
			case 'three':
				$classes[] = 'top-bar-centered';
				break;
		}

		if ( ! $has_split_bk ) {
			$classes[] = 'wpex-flex-shrink-0';
		}

		$classes[] = 'wpex-clr';

		/**
		 * Filters the topbar content element class.
		 *
		 * @param array $class
		 */
		$classes = (array) apply_filters( 'totaltheme/topbar/content_class', $classes );

		/*** deprecated ***/
		$classes = (array) apply_filters( 'wpex_top_bar_classes', $classes );
		$classes = (array) apply_filters( 'wpex_topbar_content_class', $classes );

		return implode( ' ', $classes );
	}

	/**
	 * Echo class attribute for the the topbar content element.
	 */
	public static function content_class() {
		$classes = self::get_content_classes();
		if ( $classes ) {
			echo 'class="' . esc_attr( $classes ) . '"';
		}
	}

}