<?php
namespace TotalTheme;

use TotalTheme\Integration\WooCommerce;
use TotalTheme\Replace_Vars;

defined( 'ABSPATH' ) || exit;

/**
 * Title Class.
 *
 * @package TotalTheme
 * @subpackage Page
 * @version 5.6.1
 */
class Title {

	/**
	 * Meta key where the custom title is saved.
	 */
	public const META_KEY = 'wpex_post_title';

	/**
	 * Class instance.
	 */
	private static $instance;

	/**
	 * Create or retrieve the instance of this class.
	 */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new self();
		}
		return static::$instance;
	}

	/**
	 * Checks if the header is enabled or not.
	 */
	public function get() {
		$title = null;
		$post_id = 0;

		// Single posts.
		if ( is_singular() ) {
			$post_id = get_the_ID();
			$title   = $this->get_singular_title( $post_id );
		}

		// Homepage - display blog description if not a static page.
		elseif ( is_front_page() ) {
			if ( get_bloginfo( 'description' ) ) {
				$title = get_bloginfo( 'description' );
			} else {
				$title = esc_html__( 'Recent Posts', 'total' );
			}
		}

		// Homepage posts page.
		elseif ( is_home() ) {
			$title = $this->get_posts_page_title();
		}

		// Search => NEEDS to go before archives since it's technically an archive.
		elseif ( is_search() ) {
			$title = esc_html__( 'Search results for:', 'total' ) . ' &quot;' . esc_html( get_search_query( false ) ) . '&quot;';
		}

		// Archives.
		elseif ( is_archive() ) {
			$title = $this->get_archive_title();
		}

		// 404 Page.
		elseif ( is_404() ) {

			// Custom 404 page.
			if ( $page_id = wpex_parse_obj_id( get_theme_mod( 'error_page_content_id' ), 'page' ) ) {
				$title = get_the_title( $page_id );
			}

			// Default 404 page.
			else {
				$title = wpex_get_translated_theme_mod( 'error_page_title' ) ?: esc_html__( '404: Page Not Found', 'total' );
			}

		}

		// WooCommerce titles (added here to provide support for vanilla WooCommerce).
		if ( wpex_is_woocommerce_active() ) {
			$woo_title = (new WooCommerce\Title)->get();
			if ( $woo_title ) {
				$title = $woo_title;
			}
		}

		// Check meta last.
		if ( $meta_title = $this->get_meta_title( $post_id ) ) {
			$title = $meta_title;
		}

		// Last check if title is empty.
		if ( ! $title ) {
			$post_id = wpex_get_current_post_id();
			$title   = get_the_title( $post_id );
		}

		/**
		 * Filters the current page header title text.
		 *
		 * @param string $title The title to be displayed.
		 * @param int $post_id The current post ID.
		 */
		$title = (string) apply_filters( 'wpex_title', $title, $post_id );

		$title = $this->replace_vars( $title );

		return $title;
	}

	/**
	 * Returns post specific title used for elements.
	 */
	public function get_unfiltered_post_title( $post_id = '' ) {
		return $this->get_meta_title( $post_id ) ?: get_the_title( $post_id );
	}

	/**
	 * Returns custom title.
	 */
	public function get_meta_title( $post_id = '' ) {
		$post_id = $post_id ?: wpex_get_current_post_id();
		$meta_title = get_post_meta( $post_id, self::META_KEY, true );
		if ( $meta_title ) {
			$post = get_post( $post_id );
			if ( ! empty( $post->post_password ) ) {
				$prepend = esc_html__( 'Protected: %s', 'total' );
				$protected_title_format = apply_filters( 'protected_title_format', $prepend, $post );
				return sprintf( $protected_title_format, $meta_title );
			} elseif ( isset( $post->post_status ) && 'private' === $post->post_status ) {
				$prepend = esc_html__( 'Private: %s', 'total' );
				$private_title_format = apply_filters( 'private_title_format', $prepend, $post );
				return sprintf( $private_title_format, $meta_title );
			}
			return $meta_title;
		}
	}

	/**
	 * Returns the singular title.
	 */
	protected function get_singular_title() {
		$type = get_post_type();

		switch ( $type ) {
			case 'post':
				$title = $this->get_standard_post_title();
				break;
			case 'page':
			case 'attachment':
			case 'wp_router_page':
			case 'templatera':
				$title = get_the_title();
				break;
			default:
				$title = $this->get_cpt_post_title( $type );
				break;
		}

		return $title;
	}

	/**
	 * Returns the standard post title.
	 */
	protected function get_standard_post_title() {
		$title = '';
		$display = get_theme_mod( 'blog_single_header', 'custom_text' );
		switch ( $display ) {
			case 'custom_text':
				$title = wpex_get_translated_theme_mod( 'blog_single_header_custom_text' );
				if ( ! $title ) {
					$title = esc_html__( 'Blog', 'total' );
				}
				break;
			case 'first_category':
				$title = wpex_get_first_term_name( get_post(), 'category' );
				break;
			default:
				$title = get_the_title( get_post() );
				break;
		}
		return $title;
	}

	/**
	 * Returns the custom post type post title.
	 */
	protected function get_cpt_post_title( $cpt = '' ) {
		$title = '';
		$cpt   = $cpt ?: get_post_type();

		if ( WPEX_PTU_ACTIVE ) {
			$title = wpex_get_ptu_type_mod( $cpt, 'page_header_title' );
			if ( $title && is_string( $title ) ) {
				return $title;
			}
		}

		if ( defined( 'TYPES_VERSION' ) ) {
			$title = get_theme_mod( 'cpt_single_page_header_text', null );
			if ( $title && is_string( $title ) ) {
				return $title;
			}
		}

		$display = get_theme_mod( $cpt . '_single_header' );

		switch ( $display ) {
			case 'custom_text':
				$title = wpex_get_translated_theme_mod( $cpt . '_single_header_custom_text' );
				break;
			case 'first_category':
				$title = wpex_get_first_term_name();
				break;
			case 'post_title':
				$title = get_the_title();
				break;
			default:
				$obj = get_post_type_object( $cpt );
				if ( is_object( $obj ) ) {
					$title = $obj->labels->name ?? '';
				}
				break;
		}

		return $title;
	}

	/**
	 * Returns the current archive title.
	 */
	protected function get_archive_title() {

		// Author.
		if ( is_author() ) {
			if ( $author = get_queried_object() ) {
				return $author->display_name; // Fix for authors with 0 posts
			} else {
				return get_the_archive_title();
			}
		}

		// Post Type archive title.
		elseif ( is_post_type_archive() ) {
			if ( WPEX_PTU_ACTIVE ) {
				$ptu_title = wpex_get_ptu_type_mod( get_query_var( 'post_type' ), 'archive_page_header_title' );
				if ( $ptu_title && is_string( $ptu_title ) ) {
					return $ptu_title;
				}
			}
			return post_type_archive_title( '', false );
		}

		// Daily archive title.
		elseif ( is_day() ) {
			return sprintf( esc_html__( 'Daily Archives: %s', 'total' ), get_the_date() );
		}

		// Monthly archive title.
		elseif ( is_month() ) {
			return sprintf( esc_html__( 'Monthly Archives: %s', 'total' ), get_the_date( 'F Y' ) );
		}

		// Yearly archive title.
		elseif ( is_year() ) {
			return sprintf( esc_html__( 'Yearly Archives: %s', 'total' ), get_the_date( 'Y' ) );
		}

		// Categories/Tags/Other.
		else {
			if ( WPEX_PTU_ACTIVE && is_tax() ) {
				$ptu_title = wpex_get_ptu_tax_mod( get_query_var( 'taxonomy' ), 'page_header_title' );
				if ( $ptu_title && is_string( $ptu_title ) ) {
					return $ptu_title;
				}
			}
			return single_term_title( '', false );
		}
	}

	/**
	 * Returns the posts page (home) title.
	 */
	protected function get_posts_page_title() {
		return get_the_title( get_option( 'page_for_posts', true ) ) ?: esc_html__( 'Home', 'total' );
	}

	/**
	 * Replaces variables.
	 */
	protected function replace_vars( $string =  '' ) {
		if ( ! $string ) {
			return '';
		}
		return (new Replace_Vars)->replace( $string );
	}

}