<?php
namespace TotalTheme;

defined( 'ABSPATH' ) || exit;

/**
 * Theme Icon methods.
 *
 * @package Total WordPress theme
 * @subpackage Classes
 * @version 5.6.1
 */
class Theme_Icons {

	/**
	 * Check if cache is enabled.
	 */
	const USE_CACHE = true;

	/**
	 * Check if we should search for custom child theme icons.
	 *
	 * @todo integrate?
	 */
	const GET_CHILD_THEME_ICONS = false;

	/**
	 * Font icon stylesheet name.
	 */
	const STYLE_HANDLE = 'ticons';

	/**
	 * Transient name for getting icons list.
	 */
	const TRANSIENT = 'wpex_theme_icons_list';

	/**
	 * Holds array of icons.
	 */
	protected static $icons_list = null;

	/**
	 * Checks if theme icons are enabled.
	 */
	public static function is_enabled() {
		return apply_filters( 'wpex_ticons_enable', true );
	}

	/**
	 * Returns Theme Icons CSS file.
	 */
	public static function get_css_url() {
		return wpex_asset_url( 'lib/ticons/css/ticons.min.css' );
	}

	/**
	 * Returns a json list of all Theme Icons.
	 */
	public static function get_json_url() {
		return wpex_asset_url( 'lib/ticons/json/ticons.json' );
	}

	/**
	 * Registers the theme icon font stylesheet.
	 */
	public static function register_font_style() {
		wp_register_style(
			self::STYLE_HANDLE,
			self::get_css_url(),
			[],
			WPEX_THEME_VERSION
		);
	}

	/**
	 * Enqueues theme icon font stylesheet.
	 */
	public static function enqueue_font_style() {
		wp_enqueue_style( self::STYLE_HANDLE );
	}

	/**
	 * Returns list of all available theme icons.
	 */
	public static function get_icons_list() {
		if ( ! is_null( self::$icons_list ) ) {
			return self::$icons_list;
		}

		$icons      = [ 'none' ];
		$icon_files = self::list_files();

		if ( $icon_files ) {
			$icons = array_merge( $icons, $icon_files );
		}

		// Set keys equal to vars.
		$icons = array_combine( $icons, $icons );

		// Old Filter @todo deprecate
		$icons = apply_filters( 'wpex_get_awesome_icons', $icons );

		/**
		 * Filters the list of theme icons.
		 *
		 * @param array $icons
		 */
		self::$icons_list = (array) apply_filters( 'wpex_ticons_list', $icons );

		return self::$icons_list;
	}

	/**
	 * Returns the preferred format for theme icons.
	 */
	public static function get_format() {
		$format = 'font';

		if ( get_theme_mod( 'svg_icons_enable', false ) ) {
			$format = 'svg';
		}

		/**
		 * Filters the theme icon format (font|svg).
		 *
		 * @param string $format
		 */
		$format = (string) apply_filters( 'wpex_theme_icon_format', $format );

		return $format;
	}

	/**
	 * Returns icon.
	 */
	public static function get_icon( $icon = '', $extra_class = '' ) {
		if ( ! $icon
			|| ! is_string( $icon )
			|| '' === trim( $icon )
			|| 'ticon ticon-none' === $icon
			|| 'none' === $icon
		) {
			return;
		}

		$html       = '';
		$icon_attrs = [];
		$format     = self::get_format();
		$is_ticon   = true;

		if ( false !== strpos( $icon, '/' ) && ! str_starts_with( $icon, 'ticon' ) ) {
			$is_ticon = false;
			$format = 'svg';
		//	$icon = str_replace( 'custom/', '', $icon ); // @todo?
		} else {
			$icon = str_replace( 'ticons/', '', $icon );
		}

		// Add svg class.
		if ( 'svg' === $format ) {
			$icon_attrs['class'] = [ 'wpex-svg-icon' ];
		}

		// Add ticon prefix and ticon class.
		else {
			if ( is_string( $icon ) && str_starts_with( $icon, 'ticon' ) ) {
				$class = explode( ' ', $icon );
			} else {
				$class = [ 'ticon' ];
				$class[] = 'ticon-' . sanitize_html_class( $icon );
			}
			$icon_attrs['class'] = $class;
		}

		$icon_attrs['aria-hidden'] = 'true';

		if ( $extra_class ) {
			if ( is_array( $extra_class ) ) {
				$extra_classes = $extra_class['class'] ?? null;
				unset( $extra_class['class'] );
				$icon_attrs = wp_parse_args( $extra_class, $icon_attrs );
				array_unshift( $icon_attrs['class'], $extra_classes );
			} else {
				array_unshift( $icon_attrs['class'], $extra_class );
			}
		}

		switch ( $format ) {
			case 'svg':
				$svg_path = $icon;
				if ( $is_ticon ) {
					$icon = str_replace( 'ticon-', '', $icon );
					$icon = str_replace( 'ticon', '', $icon );
					$svg_path = 'ticons/' . trim( $icon );
				}
				$get_svg = wpex_get_svg( $svg_path );
				if ( $get_svg ) {
					$html = '<span ' . trim( wpex_parse_attrs( $icon_attrs ) ) . '>' . $get_svg . '</span>';
				}
				break;
			case 'font':
			default:
				self::enqueue_font_style();
				$html = '<span ' . trim( wpex_parse_attrs( $icon_attrs ) ) . '></span>';
				break;
		}

		/**
		 * Filters the theme icon html.
		 *
		 * @param string $html
		 * @param $icon
		 * @param $extra_class
		 */
		$html = (string) apply_filters( 'wpex_theme_icon_html', $html, $icon, $extra_class );

		return $html;
	}

	/**
	 * Renders an icon.
	 */
	public static function render_icon( $icon = '', $extra_class = '' ) {
		echo self::get_icon( $icon, $extra_class );
	}

	/**
	 * Returns a list of all icon file names inside a given directory.
	 */
	protected static function get_files_list( $dir_str = '' ) {
		$files = [];
		$dir   = @opendir( $dir_str );
		if ( $dir ) {
			while ( false !== ( $file = readdir( $dir ) ) ) {
				// Skip current and parent folder links.
				if ( in_array( $file, [ '.', '..', 'ticons' ], true ) ) {
					continue;
				}
				// Skip hidden files or folders.
				if ( '.' === $file[0] ) {
					continue;
				}
				$file_path = $dir_str . DIRECTORY_SEPARATOR . $file;
				if ( is_file( $file_path ) ) {
					$info = pathinfo( $file_path );
					if ( ! empty( $info['filename'] ) && isset( $info['extension'] ) && 'svg' === $info['extension'] ) {
						$files[] = $info['filename'];
					}
				}
			}
			closedir( $dir );
		}
		return $files;
	}

	/**
	 * Returns the list of all icons.
	 */
	protected static function list_files() {
	//	self::delete_transient();
		$files = self::USE_CACHE ? get_transient( self::TRANSIENT ) : [];
		if ( $files && is_array( $files ) ) {
			return $files;
		}
		$files = self::get_files_list( trailingslashit( get_template_directory() ) . 'assets/svgs/ticons' );
		if ( self::GET_CHILD_THEME_ICONS ) {
			if ( $files && is_child_theme() ) {
				$child_files = self::list_child_theme_files();
				if ( is_array( $child_files ) ) {
					$files = array_merge( $files, $child_files );
				}
			}
		}
		if ( self::USE_CACHE ) {
			set_transient( self::TRANSIENT, $files, MONTH_IN_SECONDS );
		}
		return $files;
	}

	/**
	 * Returns the list of child theme icons.
	 */
	protected static function list_child_theme_files() {
		$child_files = self::get_files_list( trailingslashit( get_stylesheet_directory() ) . 'assets/svgs/ticons' );
		if ( $child_files ) {
			return preg_filter('/^/', 'custom/', $child_files); // adds custom/ to the front of the filename so the theme knows they aren't theme icons.
		}
	}

	/**
	 * Generates a json array of the theme icons
	 */
	protected static function generate_json_list() {
		$iconList = [];
		$icon_files = self::list_files();
		foreach ( $icon_files as $icon_file ) {
			$iconList[] = $icon_file;
		}
		sort( $iconList );
		$icons = new \stdClass();
		$icons->icons = $iconList;
		return json_encode( $icons );
	}

	/**
	 * Deletes the list transient.
	 */
	public static function delete_transient() {
		delete_transient( self::TRANSIENT );
	}

}