<?php
namespace TotalTheme;

defined( 'ABSPATH' ) || exit;

/**
 * Sanitize inputted data.
 *
 * @package TotalTheme
 * @version 5.6.1
 */
final class Sanitize_Data {

	/**
	 * Main class function parses input to return sanitized output.
	 */
	public function parse_data( $input, $type ) {
		$type = str_replace( '-', '_', $type );
		if ( method_exists( $this, $type ) ) {
			return $this->$type( $input );
		}
		return $input; // perhaps we should at least strip all tags?
	}

	/**
	 * URL.
	 */
	public function url( $input ) {
		return esc_url( $input );
	}

	/**
	 * Text.
	 */
	public function text( $input ) {
		return sanitize_text_field( $input );
	}

	/**
	 * Text Field.
	 */
	public function text_field( $input ) {
		return sanitize_text_field( $input );
	}

	/**
	 * Textarea.
	 */
	public function textarea( $input ) {
		return wp_kses_post( $input );
	}

	/**
	 * Boolean.
	 */
	public function boolean( $input ) {
		return wpex_validate_boolean( $input );
	}

	/**
	 * Pixels.
	 */
	public function px( $input ) {
		if ( 'none' === $input ) {
			return '0px';
		} else {
			return floatval( $input ) . 'px'; // Not sure why we used floatval but lets leave it incase
		}
	}

	/**
	 * Milliseconds.
	 */
	public function ms( $input ) {
		return floatval( $input ) . 'ms';
	}

	/**
	 * Pixel Fallback.
	 */
	public function fallback_px( $input ) {
		if ( 'none' === $input ) {
			return '0px';
		} elseif ( is_numeric( $input ) ) {
			return floatval( $input ) . 'px'; // Not sure why we used floatval but lets leave it incase
		}
		return wp_strip_all_tags( $input );
	}

	/**
	 * Container width.
	 */
	public function container_width( $input ) {
		if ( is_numeric( $input ) ) {
			return; // Prevent values without units from working - this is a fix for pre 5.4.
		}
		return wp_strip_all_tags( $input );
	}

	/**
	 * Margin.
	 */
	public function margin( $input ) {
		if ( 'none' === $input || '0' === $input || 0 === $input || '0px' === $input ) {
			return '0px'; // must include px to prevent issues with calc
		}

		if ( is_numeric( $input ) ) {
			return $input  . 'px';
		}

		return esc_attr( $input );
	}

	/**
	 * Padding.
	 */
	public function padding( $input ) {
		if ( 'none' === $input || '0' === $input || 0 === $input || '0px' === $input ) {
			return '0px'; // must include px to prevent issues with calc
		}

		if ( is_numeric( $input ) ) {
			return $input  . 'px';
		}

		return esc_attr( $input );
	}

	/**
	 * Font Size utl.
	 */
	public function utl_font_size( $input ) {
		if ( $input && array_key_exists( $input, wpex_utl_font_sizes() ) ) {
			return 'var(--wpex-text-' . esc_attr( $input ) . ')';
		}
	}

	/**
	 * Border Radius.
	 */
	public function utl_border_radius( $input ) {
		if ( 'rounded-0' === $input ) {
			return '0';
		}
		if ( 'rounded-full' === $input ) {
			return '9999px';
		}
		if ( $input && array_key_exists( $input, wpex_utl_border_radius() ) ) {
			return 'var(--wpex-' . esc_attr( $input ) . ')';
		}
	}

	/**
	 * Font Size.
	 */
	public function font_size( $input ) {
		return wpex_sanitize_font_size( $input, 'font_size' );
	}

	/**
	 * Font Weight.
	 */
	public function font_weight( $input ) {
		switch ( $input ) {
			case 'normal':
				return '400';
				break;
			case 'semibold':
				return '600';
				break;
			case 'bold':
				return '700';
				break;
			case 'bolder':
				return '900';
				break;
			default:
				return wp_strip_all_tags( $input );
				break;
		}
	}

	/**
	 * Hex Color.
	 */
	public function hex_color( $input ) {
		if ( ! $input ) {
			return null;
		} elseif ( 'none' === $input ) {
			return 'transparent';
		} elseif ( preg_match('|^#([A-Fa-f0-9]{3}){1,2}$|', $input ) ) {
			return $input;
		} else {
			return null;
		}
	}

	/**
	 * Border Radius.
	 */
	public function border_radius( $input ) {
		if ( 'none' === $input ) {
			return '0';
		}

		if ( is_numeric( $input ) ) {
			return $input . 'px';
		}

		return esc_attr( $input );
	}

	/**
	 * Pixel or Percent.
	 */
	public function px_pct( $input ) {
		if ( 'none' === $input || '0px' === $input ) {
			return '0';
		} elseif ( strpos( $input, '%' ) ) {
			return wp_strip_all_tags( $input );
		} elseif ( $input = floatval( $input ) ) {
			return wp_strip_all_tags( $input ) .'px';
		}
	}

	/**
	 * Pixel or Em.
	 */
	public function px_em( $input ) {
		if ( 'none' === $input || '0px' === $input) {
			return '0';
		}

		if ( is_numeric( $input ) ) {
			return $input . 'px';
		}

		$unit = $this->get_unit( $input );

		if ( 'px' === $unit || 'em' === $unit ) {
			return esc_attr( $input );
		}

		$input = floatval( $input );

		if ( $input ) {
			return $input . 'px';
		}
	}

	/**
	 * Opacity.
	 */
	public function opacity( $input ) {
		if ( ! is_numeric( $input ) || $input > 1 ) {
			return;
		} else {
			return wp_strip_all_tags( $input );
		}
	}

	/**
	 * HTML.
	 */
	public function html( $input ) {
		return wp_kses_post( $input );
	}

	/**
	 * Image.
	 *
	 * @todo remove, image markup should be trusted the attributes should be sanitized before hand.
	 */
	public function img( $input ) {
		return wp_kses( $input, [
			'img' => [
				'src'      => [],
				'alt'      => [],
				'srcset'   => [],
				'id'       => [],
				'class'    => [],
				'height'   => [],
				'width'    => [],
				'data'     => [],
				'data-rjs' => [],
			],
		] );
	}

	/**
	 * Image from setting.
	 */
	public function image_src_from_mod( $input ) {
		return wpex_get_image_url( $input );
	}

	/**
	 * Background Style.
	 */
	public function background_style_css( $input ) {
		switch ( $input ) {
			case 'stretched':
				return 'background-size: cover;
					background-position: center center;
					background-attachment: fixed;
					background-repeat: no-repeat;';
				break;
			case 'cover':
				return 'background-position:center center;background-size: cover;';
				break;
			case 'repeat':
				return 'background-repeat:repeat;';
				break;
			case 'no-repeat':
				return 'background-repeat:no-repeat;';
				break;
			case 'repeat-y':
				return 'background-position: center center;background-repeat:repeat-y;';
				break;
			case 'fixed':
				return 'background-repeat: no-repeat;background-position: center center;background-attachment: fixed;';
				break;
			case 'fixed-top':
				return 'background-repeat: no-repeat;background-position: center top;background-attachment: fixed;';
				break;
			case 'fixed-bottom':
				return 'background-repeat: no-repeat;background-position: center bottom;background-attachment: fixed;';
				break;
			default:
				return 'background-repeat:' . esc_attr( $input ) . ';';
				break;
		}

	}

	/**
	 * Embed URL.
	 */
	public function embed_url( $url ) {
		return wpex_get_video_embed_url( $url );
	}

	/**
	 * Google Map Embed.
	 */
	public function google_map( $input ) {
		return wp_kses( $input, [
			'iframe' => [
				'src'             => [],
				'height'          => [],
				'width'           => [],
				'frameborder'     => [],
				'style'           => [],
				'allowfullscreen' => [],
			],
		] );
	}

	/**
	 * iFrame.
	 */
	public function iframe( $input ) {
		return wp_kses( $input, [
			'iframe' => [
				'align' => [],
				'width' => [],
				'height' => [],
				'frameborder' => [],
				'name' => [],
				'src' => [],
				'id' => [],
				'class' => [],
				'style' => [],
				'scrolling' => [],
				'marginwidth' => [],
				'marginheight' => [],
				'allow' => [],
			],
		] );
	}

	/**
	 * SVG.
	 */
	public function svg( $input ) {
		return wp_kses( $input, [
			'svg' => [
				'class' => true,
				'aria-hidden' => true,
				'aria-labelledby' => true,
				'role' => true,
				'xmlns' => true,
				'width' => true,
				'height' => true,
				'viewbox' => true, // <= Must be lower case!
				'style' => true,
				'preserveAspectRatio' => true, // WP bug prevents this from working :(
			],
			'g' => [ 'fill' => true ],
			'title' => [ 'title' => true ],
			'path' => [
				'd' => true,
				'fill' => true,
				'class' => true,
			],
		] );
	}

	/**
	 * Return css unit (text) from input.
	 */
	protected function get_unit( $input ) {
		if ( $input && is_string( $input ) && ! is_numeric( $input ) ) {
			$non_numeric_string = preg_replace( '/[^0-9.]/', '', $input );
			$unit = str_replace( $non_numeric_string, '', $input );
			return trim( $unit );
		}
	}

}