<?php
namespace TotalTheme;

defined( 'ABSPATH' ) || exit;

/**
 * Replace_Vars Class.
 *
 * @package TotalTheme
 * @version 5.6.1
 */
final class Replace_Vars {

	/**
	 * Check if functionality is enabled.
	 *
	 * @return bool $check
	 */
	protected function is_enabled() {
		return (bool) apply_filters( 'totaltheme/replace_vars/is_enabled', true );
	}

	/**
	 * Replace `{{variable_placeholders}}` with their correct value.
	 *
	 * @param string $text The string to replace the variables in.
	 *
	 * @return string
	 */
	public function replace( $text, $args = [] ) {
		if ( ! is_string( $text )
			|| ! $this->is_enabled()
			|| false === strpos( $text, '{{' )
		) {
			return $text;
		}

		$text = preg_replace_callback( "/\{\{([^}]*)}\}/", [ $this, 'handle_replacements' ], $text );

		return $text;
	}

	/**
	 * Returns the array of variables and their values.
	 *
	 * @return array
	 */
	private function get_vars() {
		$vars = [
			'post_id'           => '',
			'post_rating'       => '',
			'post_author'       => '',
			'post_date'         => '',
			'post_modified'     => '',
			'post_title'        => '',
			'post_slug'         => '',
			'post_subheading'   => '',
			'title'             => '',
			'taxonomy'          => '',
			'term_id'           => '',
			'term_name'         => '',
			'permalink'         => '',
			'category'          => '',
			'primary_term_id'   => '',
			'paged'             => '',
			'post_count'        => '',

			// @todo ?
		//	'author_first_name' => '',
		//	'author_last_name'  => '',
		//	'post_year'         => '',
		//	'post_month'        => '',
		//	'post_day'          => '',
		//	'currentdate'       => '',
		//	'currentyear'       => '',
		//	'currentmonth'      => '',
		//	'currentday'        => '',
		//	'post_content'      => '',
		//	'post_excerpt'      => '',
		//	'max_num_pages'     => '',
		];

		/**
		 * Filters the available vars.
		 *
		 * @param array $vars List of vars.
		 */
		$vars = (array) apply_filters( 'totaltheme/replace_vars/vars', $vars );

		return $vars;
	}

	/**
	 * Get the title var value.
	 *
	 * @param array $matches The matches returned by preg_replace_callback
	 *
	 * @return string
	 */
	private function handle_replacements( $matches ) {
		$vars        = $this->get_vars();
		$replacement = $matches[0];
		$has_args    = $has_args = $this->var_has_args( $matches[1] );

		if ( array_key_exists( $matches[1], $vars ) || $has_args ) {
			if ( ! $has_args && ! empty( $vars[$matches[1]] ) ) {
				$replacement = is_callable( $vars[$matches[1]] ) ? call_user_func( $vars[$matches[1]] ) : $vars[$matches[1]];
			} else {
				$method_suffix = $has_args ? strtok( $matches[1], '_' ) : $matches[1];
				$method = 'get_' . $method_suffix;
				if ( method_exists( $this, $method ) ) {
					if ( $has_args ) {
						$replacement = $this->$method( $matches[1] );
					} else {
						$replacement = $this->$method();
					}
				}
			}
		}

		if ( is_null( $replacement ) || false === $replacement ) {
			$replacement = '';
		} elseif ( ! is_scalar( $replacement ) ) {
			$replacement = $matches[0];
		}

		return $replacement;
	}

	/**
	 * Checks if the current variable has args.
	 *
	 * @return bool
	 */
	private function var_has_args( $var = '' ) {
		if ( str_starts_with( $var, 'acf_' ) || str_starts_with( $var, 'cf_' ) ) {
			return true;
		}
	}

	/**
	 * Get the title var value.
	 *
	 * @return string
	 */
	private function get_title() {
		if ( is_tax() && ! ( in_the_loop() || get_query_var( 'wpex_card_object' ) ) ) {
			return single_term_title( '', false );
		} else {
			return $this->get_post_title();
		}
	}

	/**
	 * Get the post_title var value.
	 *
	 * @return string
	 */
	private function get_post_title() {
		return get_the_title();
	}

	/**
	 * Get the post id.
	 *
	 * @return int
	 */
	private function get_post_id() {
		return get_the_ID();
	}

	/**
	 * Get Post Rating.
	 *
	 * @return int|string
	 */
	private function get_post_rating( $var = '' ) {
		$rating = get_post_meta( get_the_ID(), 'wpex_post_rating', true );
		if ( ! $rating && function_exists( 'wc_get_product' ) && 'product' === get_post_type() ) {
			$product = wc_get_product( get_the_ID() );
			if ( $product && is_callable( [ $product, 'get_average_rating' ] ) ) {
				$rating = $product->get_average_rating();
			}
		}
		return $rating;
	}

	/**
	 * Get the category name.
	 *
	 * @return string
	 */
	private function get_category() {
		return wpex_get_first_term_name();
	}

	/**
	 * Get the primary term id.
	 *
	 * @return string
	 */
	private function get_primary_term_id() {
		return wpex_get_first_term_id();
	}

	/**
	 * Get the post date.
	 *
	 * @return string
	 */
	private function get_post_date() {
		return get_the_date();
	}

	/**
	 * Get the post modified date.
	 *
	 * @return string
	 */
	private function get_post_modified() {
		return get_the_modified_date();
	}

	/**
	 * Get post slug.
	 *
	 * @return string
	 */
	private function get_post_slug() {
		return get_post_field( 'post_name', get_post() );
	}

	/**
	 * Get the post permalink.
	 *
	 * @return string
	 */
	private function get_permalink() {
		return get_permalink();
	}

	/**
	 * Get the post author.
	 *
	 * @return string
	 */
	private function get_post_author() {
		return get_the_author();
	}

	/**
	 * Get the post subheading.
	 *
	 * @return string
	 */
	private function get_post_subheading() {
		return wpex_page_header_subheading_content();
	}

	/**
	 * Get current taxonomy name.
	 *
	 * @return string
	 */
	private function get_taxonomy() {
		$obj = get_queried_object();
		if ( is_a( $obj, 'WP_Term' ) ) {
			$tax = get_taxonomy( $obj->taxonomy );
			if ( is_a( $tax, 'WP_Taxonomy' ) ) {
				return $tax->labels->singular_name;
			}
		}
	}

	/**
	 * Get current taxonomy term id.
	 *
	 * @return string
	 */
	private function get_term_id() {
		$obj = get_queried_object();
		if ( is_a( $obj, 'WP_Term' ) ) {
			return $obj->term_id;
		}
	}

	/**
	 * Get current taxonomy term name.
	 *
	 * @return string
	 */
	private function get_term_name() {
		$obj = get_queried_object();
		if ( is_a( $obj, 'WP_Term' ) ) {
			return $obj->name;
		}
	}

	/**
	 * Get post count.
	 *
	 * @return string
	 */
	private function get_post_count() {
		global $wp_query;
		return $wp_query->found_posts ?? '';
	}

	/**
	 * Get paged text.
	 *
	 * @return string
	 */
	private function get_paged() {
		if ( ! is_paged() ) {
			return '';
		}
		$paged = get_query_var( 'paged' ) ?: get_query_var( 'page' ) ?: 1;
		if ( $paged > 1 ) {
			global $wp_query;
			$max_num_pages = $wp_query->max_num_pages ?? 0;
			if ( $max_num_pages ) {
				return sprintf( esc_html__( 'Page %s of %s' ), $paged, $max_num_pages );
			} else {
				return sprintf( esc_html__( 'Page %s' ), $paged );
			}
		}
	}

	/**
	 * Get custom field value.
	 *
	 * @return string
	 */
	private function get_cf( $var = '' ) {
		$key = str_replace( 'cf_', '', $var );
		if ( $key ) {
			if ( function_exists( 'get_field' ) ) {
				return get_field( $key ) ?: '';
			}
			return get_post_meta( get_the_ID(), $key, true );
		}
	}

	/**
	 * Get acf field value.
	 *
	 * @return string
	 */
	private function get_acf( $var = '' ) {
		$key = str_replace( 'acf_', '', $var );
		if ( $key ) {
			if ( ! function_exists( 'get_field' ) ) {
				return '{{' . $var . '}}';
			}
			return get_field( $key ) ?: '';
		}
	}

}