<?php
namespace TotalTheme;

defined( 'ABSPATH' ) || exit;

/**
 * Local Scroll.
 *
 * @package TotalTheme
 * @version 5.6.1
 */
class Local_Scroll {

	/**
	 * Holds list of enabled features.
	 */
	protected $enabled_features;

	/**
	 * Instance.
	 *
	 * @access private
	 * @var object Class object.
	 */
	private static $instance;

	/**
	 * Create or retrieve the instance of our class.
	 */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new self();
			static::$instance->get_enabled_features();
		}
		return static::$instance;
	}

	/**
	 * Returns list of local scroll features.
	 */
	protected function get_enabled_features() {
		if ( ! is_null( $this->enabled_features ) ) {
			return $this->enabled_features;
		}

		if ( get_theme_mod( 'local_scroll_find_links', true ) ) {
			$this->enabled_features[] = 'find_links';
		}

		if ( $this->is_scroll_to_hash_supported() ) {
			$this->enabled_features[] = 'scroll_to_hash';
		}

		if ( $this->is_highlight_supported() ) {
			$this->enabled_features[] = 'highlight';
		}

		if ( $this->is_update_hash_supported() ) {
			$this->enabled_features[] = 'update_hash';
		}

		return $this->enabled_features;
	}

	/**
	 * Checks if a certain feature is enabled.
	 */
	public function is_feature_enabled( $feature = '' ) {
		return in_array( $feature, $this->get_enabled_features() );
	}

	/**
	 * Checks if scroll to hash is user enabled.
	 */
	protected function is_scroll_to_hash_supported() {
		$check = wp_validate_boolean( get_theme_mod( 'scroll_to_hash', true ) );

		/**
		 * Check if the local scroll on load function is enabled.
		 *
		 * @param bool $check
		 */
		 $check = (bool) apply_filters( 'wpex_has_local_scroll_on_load', $check );

		return $check;
	}

	/**
	 * Checks if local scroll highlighting is user enabled.
	 */
	protected function is_highlight_supported() {
		$check = wp_validate_boolean( get_theme_mod( 'local_scroll_highlight', true ) );

		/**
		 * Check if the local scroll menu item highlight function is enabled.
		 *
		 * @param bool $check
		 */
		$check = (bool) apply_filters( 'wpex_has_local_scroll_menu_highlight', $check );

		return $check;
	}

	/**
	 * Checks if scroll to hash is user enabled.
	 */
	protected function is_update_hash_supported() {
		$check = wp_validate_boolean( get_theme_mod( 'local_scroll_update_hash', false ) );

		/**
		 * Check if the local scroll hash update function is enabled.
		 *
		 * @param bool $check
		 */
		$check = (bool) apply_filters( 'wpex_has_local_scroll_hash_update', $check );

		return $check;
	}

	/**
	 * Returns time to wait when scrolling to a local section on load.
	 */
	public function get_onload_timeout_time() {
		$timeout = get_theme_mod( 'scroll_to_hash_timeout' ) ?: 500;

		/**
		 * Filters the local scroll on load timeout time.
		 *
		 * @param int $timeout
		 */
		$timeout = apply_filters( 'wpex_local_scroll_on_load_timeout', $timeout );

		return absint( $timeout );
	}

	/**
	 * Returns local scroll targets.
	 */
	public function get_trigger_targets() {
		$targets = 'li.local-scroll a, a.local-scroll, .local-scroll-link, .local-scroll-link > a';

		/**
		 * Filters the local scroll target elements.
		 *
		 * @param string $targets
		 */
		$targets = (string) apply_filters( 'wpex_local_scroll_targets', $targets );

		return wp_strip_all_tags( $targets );
	}

	/**
	 * Returns local scroll speed function.
	 */
	public function get_scroll_to_duration() {
		$speed = get_theme_mod( 'local_scroll_speed' );
		$speed = ( $speed || '0' === $speed ) ? absint( $speed ) : 1000;

		/**
		 * Filters the local scroll speed when using easing js.
		 *
		 * @param bool $check
		 */
		$speed = apply_filters( 'wpex_local_scroll_speed', $speed );

		return absint( $speed );
	}

	/**
	 * Returns local scroll behavior.
	 */
	public function get_scroll_to_behavior() {
		$behavior = get_theme_mod( 'local_scroll_behaviour' ) ?: 'smooth';

		/**
		 * Filters the local scroll speed when using easing js.
		 *
		 * @param bool $check
		 */
		$behavior = (string) apply_filters( 'wpex_local_scroll_behavior', $behavior );

		return wp_strip_all_tags( $behavior );
	}

	/**
	 * Returns easing function.
	 */
	public function get_easing_function() {
		$easing = null;

		if ( get_theme_mod( 'scroll_to_easing', false ) ) {
		   $easing = 'easeInOutExpo';
		}

		/**
		 * Filters the local scroll easing value.
		 *
		 * @param string $easing
		 */
		$easing = (string) apply_filters( 'wpex_local_scroll_easing', $easing );

		return wp_strip_all_tags( $easing );
	}

	/**
	 * Returns l10n for wp_localize_script.
	 */
	public function get_l10n() {
		$l10n = [
			'scrollToHash'          => $this->is_feature_enabled( 'scroll_to_hash' ),
			'localScrollFindLinks'  => $this->is_feature_enabled( 'find_links' ),
			'localScrollHighlight'  => $this->is_feature_enabled( 'highlight' ),
			'localScrollUpdateHash' => $this->is_feature_enabled( 'update_hash' ),
			'scrollToHashTimeout'   => $this->get_onload_timeout_time(),
			'localScrollTargets'    => $this->get_trigger_targets(),
			'localScrollSpeed'      => $this->get_scroll_to_duration(),
			'scrollToBehavior'      => $this->get_scroll_to_behavior(),
		];

		if ( $local_scroll_easing = $this->get_easing_function() ) {
			$l10n['localScrollEasing'] = esc_attr( $local_scroll_easing );
		}

		return $l10n;
	}

}