<?php
use TotalTheme\Integration\WPBakery\Helpers as WPBakery_Helpers;

defined( 'ABSPATH' ) || exit;

/**
 * WPBakery Section Configuration.
 *
 * @package TotalTheme
 * @subpackage WPBakery
 * @version 5.6.1
 */
if ( ! class_exists( 'VCEX_VC_Section_Config' ) ) {

	class VCEX_VC_Section_Config {

		/**
		 * Main constructor
		 */
		public function __construct() {
			add_action( 'vc_after_init', __CLASS__ . '::add_params', 40 ); // add params first
			add_action( 'vc_after_init', __CLASS__ . '::modify_params', 40 ); // priority is crucial.
			add_filter( 'shortcode_atts_vc_section', __CLASS__ . '::parse_attributes', 99 );
			add_filter( 'wpex_vc_section_wrap_atts', __CLASS__ . '::wrap_attributes', 10, 2 );
			add_filter( VC_SHORTCODE_CUSTOM_CSS_FILTER_TAG, __CLASS__ . '::shortcode_classes', 10, 3 );
			add_filter( 'vc_shortcode_output', __CLASS__ . '::custom_output', 10, 4 );
		}

		/**
		 * Add custom params.
		 */
		public static function add_params() {
			if ( ! function_exists( 'vc_add_params' ) ) {
				return;
			}

			$custom_params = array(
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Access', 'total' ),
					'param_name' => 'vcex_user_access',
					'weight' => 99,
					'value' => WPBakery_Helpers::get_user_access_choices(),
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Custom Access', 'total' ),
					'param_name' => 'vcex_user_access_callback',
					'value' => WPBakery_Helpers::get_user_access_custom_choices(),
					'description' => sprintf( esc_html__( 'Custom Access functions must be %swhitelisted%s for security reasons.', 'total' ), '<a href="https://total.wpexplorer.com/docs/how-to-whitelist-callback-functions-for-elements/" target="_blank" rel="noopener noreferrer">', '</a>' ),
					'weight' => 99,
					'dependency' => array( 'element' => 'vcex_user_access', 'value' => 'custom' ),
				),
				array(
					'type' => 'vcex_select',
					'heading' => esc_html__( 'Visibility', 'total' ),
					'param_name' => 'visibility',
					'weight' => 99,
				),
				array(
					'type' => 'vcex_ofswitch',
					'heading' => esc_html__( 'Use Featured Image as Background?', 'total' ),
					'param_name' => 'wpex_post_thumbnail_bg',
					'std' => 'false',
					'description' => esc_html__( 'Enable this option to use the current post featured image as the row background.', 'total' ),
					'group' => esc_html__( 'Design Options', 'total' ),
					'weight' => -2,
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__( 'Fixed Background Style', 'total' ),
					'param_name' => 'wpex_fixed_bg',
					'group' => esc_html__( 'Design Options', 'total' ),
					'weight' => -2,
					'dependency' => array( 'element' => 'parallax', 'is_empty' => true ),
					'description' => esc_html__( 'Note: Fixed backgrounds are disabled on devices under 1080px to prevent issues with mobile devices that don\'t properly support them', 'total' ),
					'value' => array(
						esc_html__( 'None', 'total' ) => '',
						esc_html__( 'Fixed', 'total' ) => 'fixed',
						esc_html__( 'Fixed top', 'total' ) => 'fixed-top',
						esc_html__( 'Fixed bottom', 'total' ) => 'fixed-bottom',
					),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Background Position', 'total' ),
					'param_name' => 'wpex_bg_position',
					'group' => esc_html__( 'Design Options', 'total' ),
					'description' => esc_html__( 'Enter your custom background position.', 'total' ) . ' (<a href="https://developer.mozilla.org/en-US/docs/Web/CSS/background-position" target="_blank" rel="noopener noreferrer">' . esc_html__( 'see mozilla docs', 'total' ) . '</a>)',
					'weight' => -2,
					'dependency' => array( 'element' => 'parallax', 'is_empty' => true ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Background Image Size', 'total' ),
					'param_name' => 'wpex_bg_size',
					'group' => esc_html__( 'Design Options', 'total' ),
					'description' => esc_html__( 'Specify the size of the background image.', 'total' ) . ' (<a href="https://developer.mozilla.org/en-US/docs/Web/CSS/background-size" target="_blank" rel="noopener noreferrer">' . esc_html__( 'see mozilla docs', 'total' ) . '</a>)',
					'weight' => -2,
					'dependency' => array( 'element' => 'parallax', 'is_empty' => true ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Z-Index', 'total' ),
					'param_name' => 'wpex_zindex',
					'group' => esc_html__( 'Design Options', 'total' ),
					'description' => esc_html__( 'Note: Adding z-index values on rows containing negative top/bottom margins will allow you to overlay the rows, however, this can make it hard to access the page builder tools in the frontend editor and you may need to use the backend editor to modify the overlapped rows.', 'total' ),
					'weight' => -2,
					'dependency' => array( 'element' => 'parallax', 'is_empty' => true ),
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Local Scroll ID', 'total' ),
					'param_name' => 'local_scroll_id',
					'description' => esc_html__( 'Unique identifier for local scrolling links.', 'total' ),
					'weight' => 99,
				),
				array(
					'type' => 'textfield',
					'heading' => esc_html__( 'Minimum Height', 'total' ),
					'description' => esc_html__( 'Adds a minimum height to the row so you can have a row without any content but still display it at a certain height. Such as a background with a video or image background but without any content.', 'total' ),
					'param_name' => 'min_height',
				),
			);

			vc_add_params( 'vc_section', $custom_params );
		}

		/**
		 * Modify core params.
		 */
		public static function modify_params() {
			if ( ! function_exists( 'vc_update_shortcode_param' ) ) {
				return;
			}

			// Move el_id.
			$param = \WPBMap::getParam( 'vc_section', 'el_id' );
			if ( $param ) {
				$param['weight'] = 99;
				vc_update_shortcode_param( 'vc_section', $param );
			}

			// Move el_class.
			$param = \WPBMap::getParam( 'vc_section', 'el_class' );
			if ( $param ) {
				$param['weight'] = 99;
				vc_update_shortcode_param( 'vc_section', $param );
			}

			// Move css_animation.
			$param = \WPBMap::getParam( 'vc_section', 'css_animation' );
			if ( $param ) {
				$param['weight'] = 99;
				vc_update_shortcode_param( 'vc_section', $param );
			}

			// Move full_width.
			$param = \WPBMap::getParam( 'vc_section', 'full_width' );
			if ( $param ) {
				$param['weight'] = 99;
				vc_update_shortcode_param( 'vc_section', $param );
			}

			// Move css.
			$param = \WPBMap::getParam( 'vc_section', 'css' );
			if ( $param ) {
				$param['group'] = esc_html__( 'Design Options', 'total' );
				$param['weight'] = -1;
				vc_update_shortcode_param( 'vc_section', $param );
			}
		}

		/**
		 * Parse VC section attributes on front-end.
		 */
		public static function parse_attributes( $atts ) {
			if ( ! empty( $atts['full_width'] )
				&& apply_filters( 'wpex_boxed_layout_vc_stretched_rows_reset', true )
				&& 'boxed' === wpex_site_layout()
			) {
				$atts['full_width'] = '';
				$atts['full_width_boxed_layout'] = 'true';
			}
			return $atts;
		}

		/**
		 * Add custom attributes to the row wrapper.
		 */
		public static function wrap_attributes( $wrapper_attributes, $atts ) {
			$inline_style = '';

			// Local scroll ID.
			if ( ! empty( $atts['local_scroll_id'] ) ) {
				$wrapper_attributes[] = 'data-ls_id="#' . esc_attr( $atts['local_scroll_id'] ) . '"';
			}

			// Min Height.
			if ( ! empty( $atts['min_height'] ) ) {

				// Sanitize min-height value.
				$min_height = $atts['min_height'];
				if ( ! preg_match('/[A-z]/', $min_height ) && strpos( $min_height, '%' ) === false ) {
					$min_height = intval( $min_height ) . 'px';
				}

				// Add min-height inline style.
				if ( $min_height ) {
					$inline_style .= 'min-height:' . esc_attr( $min_height ) . ';';
				}

			}

			// Z-Index.
			if ( ! empty( $atts['wpex_zindex'] ) ) {
				$inline_style .= 'z-index:' . esc_attr( $atts['wpex_zindex'] ) . '!important;';
			}

			// Custom background.
			if ( isset( $atts['wpex_post_thumbnail_bg'] ) && 'true' == $atts['wpex_post_thumbnail_bg'] ) {
				$thumbnail_id = WPBakery_Helpers::get_post_thumbnail_id();
				if ( $thumbnail_id ) {
					$inline_style .= 'background-image:url(' . esc_url( wp_get_attachment_image_url( $thumbnail_id, 'full' ) ) . ')!important;';
				}
			}

			// Background position.
			if ( ! empty( $atts['wpex_bg_position'] ) ) {
				$inline_style .= 'background-position:'. esc_attr( $atts['wpex_bg_position'] ) .' !important;';
			}

			// Background size.
			if ( ! empty( $atts['wpex_bg_size'] ) ) {
				$inline_style .= 'background-size:'. esc_attr( $atts['wpex_bg_size'] ) .' !important;';
			}

			// Add inline style to wrapper attributes.
			if ( $inline_style ) {
				$wrapper_attributes[] = 'style="' . esc_attr( $inline_style ) . '"';
			}

			return $wrapper_attributes;
		}

		/**
		 * Tweak shortcode classes.
		 */
		public static function shortcode_classes( $class_string, $tag, $atts ) {
			if ( 'vc_section' !== $tag ) {
				return $class_string;
			}

			$add_classes = [];

			if ( false !== strpos( $class_string, 'vc_section-has-fill' ) ) {
				$class_string = str_replace( 'vc_section-has-fill', '', $class_string );
				$add_classes['wpex-vc_section-has-fill'] = 'wpex-vc_section-has-fill';
			} elseif ( ! empty( $atts['vcex_parallax'] ) || ! empty( $atts['wpex_self_hosted_video_bg'] ) ) {
				$add_classes['wpex-vc_section-has-fill'] = 'wpex-vc_section-has-fill';
			}

			if ( ! empty( $atts['visibility'] ) ) {
				$add_classes[] = wpex_visibility_class( $atts['visibility'] );
			}

			if ( ! empty( $atts['full_width'] ) ) {
				$add_classes[] = 'wpex-vc-row-stretched';
			}

			if ( ! empty( $atts['full_width_boxed_layout'] ) ) {
				$add_classes[] = 'wpex-vc-section-boxed-layout-stretched';
			}

			if ( empty( $atts['full_width'] ) && isset( $add_classes['wpex-vc_section-has-fill'] ) ) {
				$add_classes[] = 'wpex-vc-reset-negative-margin';
			}

			if ( ! empty( $atts['wpex_fixed_bg'] ) ) {
				$add_classes[] = 'bg-' . esc_attr( $atts['wpex_fixed_bg'] );
			}

			if ( $add_classes ) {
				$add_classes = array_filter( $add_classes, 'sanitize_html_class' );
				$add_classes = array_filter( $add_classes, 'trim' );
				$class_string .= ' ' . implode( ' ', $add_classes );
			}

			$class_string = (new TotalTheme\Replace_Vars)->replace( $class_string );

			return $class_string;
		}

		/**
		 * Custom HTML output.
		 */
		public static function custom_output( $output, $obj, $atts, $shortcode ) {
			if ( 'vc_section' !== $shortcode ) {
				return $output;
			}

			if ( ! WPBakery_Helpers::shortcode_has_access( $atts ) ) {
				return;
			}

			return $output;
		}

	}

}
new VCEX_VC_Section_Config();