<?php
defined( 'ABSPATH' ) || exit;

/**
 * WPBakery Text Block Configuration.
 *
 * @package TotalTheme
 * @subpackage WPBakery
 * @version 5.6.1
 */
if ( ! class_exists( 'VCEX_VC_Column_Text_Config' ) ) {

	final class VCEX_VC_Column_Text_Config {

		/**
		 * Main constructor.
		 */
		public function __construct() {
			add_action( 'init', __CLASS__ . '::add_params', 40 );
			add_action( 'vc_after_init', __CLASS__ . '::modify_params', 40 );
			add_action( 'vc_after_mapping', __CLASS__ . '::vc_after_mapping' );
			add_filter( 'vc_shortcode_output', __CLASS__ . '::custom_output', 10, 4 );
		}

		/**
		 * Adds new params for the VC Rows.
		 *
		 * @todo update to use vc_add_params instead of vc_add_param.
		 */
		public static function add_params() {

			// Visibility
			vc_add_param( 'vc_column_text', array(
				'type' => 'vcex_select',
				'heading' => esc_html__( 'Visibility', 'total' ),
				'param_name' => 'visibility',
			) );

			// Align
			vc_add_param( 'vc_column_text', array(
				'type' => 'vcex_text_alignments',
				'heading' => esc_html__( 'Align', 'total' ),
				'param_name' => 'align',
				'dependency' => array( 'element' => 'width', 'not_empty' => true ),
			) );

			// Width
			vc_add_param( 'vc_column_text', array(
				'type' => 'textfield',
				'heading' => esc_html__( 'Width', 'total' ),
				'param_name' => 'width',
				'description' => esc_html__( 'Enter a custom width instead of using breaks to slim down your content width. ', 'total' ),
			) );

			// Typography
			if ( function_exists( 'vcex_inline_style' ) ) {
				$typo_params = array(
					array(
						'type' => 'vcex_text_alignments',
						'heading' => esc_html__( 'Text Align', 'total' ),
						'param_name' => 'text_align',
					),
					array(
						'type' => 'vcex_colorpicker',
						'heading' => esc_html__( 'Color', 'total' ),
						'param_name' => 'color',
					),
					array(
						'type' => 'vcex_ofswitch',
						'std' => 'true',
						'heading' => esc_html__( 'Apply Color To Everything?', 'total' ),
						'description' => esc_html__( 'If enabled the custom color will be applied to all child elements of the text block including headings and links.', 'total' ),
						'param_name' => 'child_inherit_color',
					//	'dependency' => array( 'element' => 'color', 'not_empty' => true ), // causes vc error
					),
					array(
						'type' => 'vcex_font_size',
						'heading' => esc_html__( 'Font Size', 'total' ),
						'param_name' => 'font_size',
					),
					array(
						'type' => 'vcex_font_family_select',
						'heading' => esc_html__( 'Font Family', 'total' ),
						'param_name' => 'font_family',
					),
					array(
						'type' => 'dropdown',
						'std' => 'false',
						'heading' => esc_html__( 'Font Style', 'total' ),
						'param_name' => 'italic',
						'value' => array(
							esc_html__( 'Normal', 'total' ) => 'false',
							esc_html__( 'Italic', 'total' ) => 'true',
						),
					),
					array(
						'type' => 'vcex_preset_textfield',
						'heading' => esc_html__( 'Line Height', 'total' ),
						'param_name' => 'line_height',
						'choices' => 'line_height',
					),
					array(
						'type' => 'vcex_preset_textfield',
						'heading' => esc_html__( 'Letter Spacing', 'total' ),
						'param_name' => 'letter_spacing',
						'choices' => 'letter_spacing',
					),
					array(
						'type' => 'vcex_select',
						'heading' => esc_html__( 'Font Weight', 'total' ),
						'param_name' => 'font_weight',
					),
					array(
						'type' => 'vcex_select',
						'heading' => esc_html__( 'Text Transform', 'total' ),
						'param_name' => 'text_transform',
						'group' => esc_html__( 'Typography', 'total' ),
					),
					array(
						'type' => 'vcex_min_max',
						'heading' => esc_html__( 'Min-Max Font Size', 'total' ),
						'param_name' => 'responsive_text_min_max',
						'unit' => 'px',
						'description' => esc_html__( 'This setting allows you to define a minimum and maximum font size in pixels. Javascript will then be used to calculate an ideal font size for your text. Important: This setting works independently and will override any other predefined font size and is recommend only for very large banners/headings.', 'total' ),
						'group' => esc_html__( 'Typography', 'total' ),
					),
					// Deprecated params
					array( 'type' => 'hidden', 'param_name' => 'responsive_text', 'std' => '' ),
					array( 'type' => 'hidden', 'param_name' => 'min_font_size', 'std' => '' ),
				);

				foreach ( $typo_params as $param ) {
					$param['group'] = esc_html__( 'Typography', 'total' );
					vc_add_param( 'vc_column_text', $param );
				}

			}

		}

		/**
		 * Modify Params.
		 */
		public static function modify_params() {
			if ( ! function_exists( 'vc_update_shortcode_param' ) ) {
				return;
			}

			// Modify css.
			$param = \WPBMap::getParam( 'vc_column_text', 'css' );
			if ( $param ) {
				$param['weight'] = -1;
				vc_update_shortcode_param( 'vc_column_text', $param );
			}
		}

		/**
		 * Parses deprecated params.
		 */
		public static function vc_after_mapping() {
			if ( function_exists( 'vc_post_param' ) && 'vc_edit_form' === vc_post_param( 'action' ) ) {
				add_filter( 'vc_edit_form_fields_attributes_vc_column_text', __CLASS__ . '::edit_fields' );
			}
		}

		/**
		 * Edit form fields.
		 */
		public static function edit_fields( $atts ) {
			if ( isset( $atts['responsive_text'] )
				&& 'true' == $atts['responsive_text']
				&& ! empty( $atts['font_size'] )
				&& ! empty( $atts['min_font_size'] )
				&& function_exists( 'vcex_parse_min_max_text_font_size' )
			) {
				$min = vcex_parse_min_max_text_font_size( $atts['min_font_size'] );
				$max = vcex_parse_min_max_text_font_size( $atts['font_size'] );
				if ( $min && $max ) {
					$atts['responsive_text_min_max'] = wp_strip_all_tags( $min . '|' . $max );
					$atts['min_font_size'] = '';
					$atts['font_size'] = '';
					$atts['responsive_text'] = '';
				}
			}

			return $atts;
		}

		/**
		 * Add custom HTML to ouput.
		 */
		public static function custom_output( $output, $obj, $atts, $shortcode ) {
			if ( 'vc_column_text' !== $shortcode ) {
				return $output;
			}

			$add_attrs = '';
			$add_classes = [];
			$inline_css = '';
			$unique_classname = function_exists( 'vcex_element_unique_classname' ) ? vcex_element_unique_classname() : '';

			// Min-Max font size.
			if ( ! empty( $atts['responsive_text_min_max'] )
				|| ( isset( $atts['responsive_text'] ) && 'true' == $atts['responsive_text'] ) // old setting
			) {

				$min_max = $atts['responsive_text_min_max'] ?? '';
				if ( $min_max && is_string( $min_max ) ) {
					$min_max = explode( '|', $min_max );
				}
				$font_size = $min_max[1] ?? $atts['font_size'] ?? null;
				$min_font_size = $min_max[0] ?? $atts['min_font_size'] ?? null;

				if ( $font_size && $min_font_size && function_exists( 'vcex_parse_min_max_text_font_size' ) ) {

					// Parse font sizes.
					$font_size     = vcex_parse_min_max_text_font_size( $font_size );
					$min_font_size = vcex_parse_min_max_text_font_size( $min_font_size );

					// Add wrap classes and data.
					if ( $font_size && $min_font_size ) {
						$add_classes[] = 'wpex-responsive-txt';
						$add_attrs .= ' data-max-font-size="' . absint( $font_size ) . '"';
						$add_attrs .= ' data-min-font-size="' . absint( $min_font_size ) . '"';
						wp_enqueue_script( 'vcex-responsive-text' );
						$atts['font_size'] = $font_size;
					}

				}

			}

			// Inline style (must run after min-max font size).
			if ( function_exists( 'vcex_inline_style' ) ) {
				$inline_style = vcex_inline_style( array(
					'color'          => $atts['color'] ?? null,
					'font_family'    => $atts['font_family'] ?? null,
					'font_size'      => $atts['font_size'] ?? null,
					'letter_spacing' => $atts['letter_spacing'] ?? null,
					'font_weight'    => $atts['font_weight'] ?? null,
					'text_align'     => $atts['text_align'] ?? null,
					'line_height'    => $atts['line_height'] ?? null,
					'width'          => $atts['width'] ?? null,
					'font_style'     => ( isset( $atts['italic'] ) && 'true' == $atts['italic'] ) ? 'italic' : '',
					'text_transform' => $atts['text_transform'] ?? null,
				), false );
				if ( $inline_style ) {
					$add_attrs .= ' style="' . $inline_style . '"';
				}
			}

			// load custom fonts.
			if ( ! empty( $atts['font_family'] ) ) {
				wpex_enqueue_google_font( $atts['font_family'] );
			}

			// Responsive CSS.
			if ( $unique_classname && function_exists( 'vcex_element_responsive_css' ) ) {

				$el_responsive_styles = array(
					'font_size' => $atts['font_size'] ?? null,
				);

				$responsive_css = vcex_element_responsive_css( $el_responsive_styles, $unique_classname );

				if ( $responsive_css ) {
					$inline_css .= $responsive_css;
					$add_classes[] = $unique_classname;
				}

			} elseif ( function_exists( 'vcex_get_module_responsive_data' )
				&& $responsive_data = vcex_get_module_responsive_data( $atts )
			) {
				$add_attrs .= ' ' . $responsive_data;
			}

			// Custom color.
			if ( ! empty( $atts['color'] ) ) {
				$child_inherit_color = wp_validate_boolean( $atts['child_inherit_color'] ?? true );
				if ( $child_inherit_color ) {
					$custom_color_classes = 'has-custom-color wpex-child-inherit-color';
					$custom_color_classes = apply_filters( 'wpex_vc_column_text_custom_color_classes', $custom_color_classes );
					if ( $custom_color_classes && is_string( $custom_color_classes ) ) {
						$add_classes[] = $custom_color_classes;
					}
				}
			}

			// Visibility.
			if ( ! empty( $atts['visibility'] ) ) {
				$add_classes[] = wpex_visibility_class( $atts['visibility'] );
			}

			// Align.
			if ( ! empty( $atts['width'] ) ) {
				$add_classes[] = 'wpex-max-w-100';
				if ( !  empty( $atts['align'] ) ) {
					$add_classes[] = 'align' . sanitize_html_class( $atts['align'] );
				} else {
					$add_classes[] = 'wpex-mx-auto';
				}
			}

			// Add classes.
			if ( $add_classes ) {
				$add_classes = implode( ' ', $add_classes );
				$output = str_replace( 'wpb_text_column', 'wpb_text_column ' . $add_classes, $output );
			}

			// Add custom attributes in first div.
			if ( $add_attrs ) {
				$pos = strpos( $output, '<div' );
				if ( $pos !== false ) {
					$output = substr_replace( $output, '<div ' . trim( $add_attrs ), $pos, strlen( '<div' ) );
				}
			}

			// Add inline CSS.
			if ( $inline_css ) {
				$inline_css = '<style>' . esc_attr( $inline_css ) . '</style>';
				$output = $inline_css . $output;
			}

			$output = (new TotalTheme\Replace_Vars)->replace( $output );

			return $output;
		}

	}

}
new VCEX_VC_Column_Text_Config();