<?php
namespace TotalTheme\Integration;

use TotalTheme\CPT\Entry_Blocks;
use TotalTheme\CPT\Single_Blocks;

defined( 'ABSPATH' ) || exit;

/**
 * Adds custom options to the Post Types Unlimited Plugin meta options.
 *
 * @package TotalTheme
 * @subpackage Integration
 * @version 5.6
 */
final class Post_Types_Unlimited {

	/**
	 * Instance.
	 */
	private static $instance;

	/**
	 * Create or retrieve the instance of Post_Types_Unlimited.
	 */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new self();
		}
		return static::$instance;
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->global_hooks();

		if ( wpex_is_request( 'admin' ) ) {
			$this->admin_hooks();
		}
	}

	/**
	 * Hook into global actions and filters.
	 */
	public function global_hooks() {
		add_filter( 'wpex_image_sizes', __CLASS__ . '::wpex_image_sizes', 100 );
		add_filter( 'wpex_register_sidebars_array', __CLASS__ . '::wpex_register_sidebars_array' );
		add_action( 'init', __CLASS__ . '::register_post_series' );
	}

	/**
	 * Hook into admin actions and filters.
	 */
	public function admin_hooks() {
		$ptu_version = defined( 'PTU_VERSION' ) ? PTU_VERSION : null;

		// Add custom metaboxes to the ptu plugin admin screen.
		if ( $ptu_version && version_compare( $ptu_version, '1.0.8', '>=' ) ) {
			add_filter( 'ptu_metaboxes', __CLASS__ . '::add_meta', 50 );
		}

		// hook into theme filters.
		add_filter( 'wpex_main_metaboxes_post_types', __CLASS__ . '::metabox_main' );
		add_filter( 'wpex_card_metabox_post_types', __CLASS__ . '::metabox_card' );
		add_filter( 'wpex_metabox_array', __CLASS__ . '::metabox_media' );
		add_filter( 'wpex_image_sizes_tabs', __CLASS__ . '::wpex_image_sizes_tabs', 50 );
		add_filter( 'wpex_gallery_metabox_post_types', __CLASS__ . '::wpex_gallery_metabox_post_types' );
		add_filter( 'wpex_dashboard_thumbnails_post_types', __CLASS__ . '::wpex_dashboard_thumbnails_post_types' );
	}

	/**
	 * Add new meta options.
	 */
	public static function add_meta( $metaboxes ) {

		/*** Post Type | General ***/
		$metaboxes[] = self::type_general_metabox();

		/*** Post Type | Archives ***/
		$metaboxes[] = self::type_archive_metabox();

		/*** Post Type | Single ***/
		$metaboxes[] = self::type_single_metabox();

		/*** Post Type | Related ***/
		$metaboxes[] = self::type_related_metabox();

		/*** Taxonomy ***/
		$metaboxes[] = self::tax_metabox();

		return $metaboxes;

	}

	/**
	 * Post Type general options.
	 */
	public static function type_general_metabox() {
		return [
			'id'       => 'total_ptu',
			'title'    => esc_html__( 'Theme Settings', 'total' ) . ' - ' . esc_html__( 'General', 'total' ),
			'screen'   => [ 'ptu' ],
			'context'  => 'normal',
			'priority' => 'low',
			'fields'   => [
				[
					'name' => esc_html__( 'Theme Settings Metabox', 'total' ),
					'id'   => 'total_ps_meta',
					'type' => 'checkbox',
				],
				[
					'name' => esc_html__( 'Metabox Media Tab', 'total' ),
					'id'   => 'total_ps_meta_media',
					'type' => 'checkbox',
				],
				[
					'name' => esc_html__( 'Card Settings Metabox', 'total' ),
					'id'   => 'total_ps_meta_card',
					'type' => 'checkbox',
				],
				[
					'name' => esc_html__( 'Post Series', 'total' ),
					'id'   => 'total_post_series',
					'type' => 'checkbox',
				],
				[
					'name' => esc_html__( 'Image Gallery', 'total' ),
					'id'   => 'total_post_gallery',
					'type' => 'checkbox',
				],
				[
					'name' => esc_html__( 'Admin Thumbnails', 'total' ),
					'id'   => 'total_show_admin_thumbnails',
					'type' => 'checkbox',
					'desc' => esc_html__( 'Check to display your post featured images on the main admin edit screen.', 'total' ),
				],
				[
					'name' => esc_html__( 'Image Sizes', 'total' ),
					'id'   => 'total_image_sizes',
					'type' => 'checkbox',
					'desc' => esc_html__( 'Enable image size settings for this post type under Theme Panel > Image Sizes.', 'total' ),
				],
				[
					'name' => esc_html__( 'Custom Sidebar', 'total' ),
					'id'   => 'total_custom_sidebar',
					'type' => 'text',
					'desc' => esc_html__( 'Enter a name to create a custom sidebar for the post type archive, single posts and attached taxonomies. The sidebar name can\'t contain any spaces and must use letters only. This is an older option and we recommend instead creating a custom widget area via Appearance > Widget Areas for your post type.', 'total' ),
				],
				[
					'name'    => esc_html__( 'Main Page', 'total' ),
					'id'      => 'total_main_page',
					'type'    => 'select',
					'desc'    => esc_html__( 'Used for breadcrumbs.', 'total' ),
					'choices' => __CLASS__ . '::choices_pages',
				],
				[
					'name'    => esc_html__( 'Main Taxonomy', 'total' ),
					'id'      => 'total_main_taxonomy',
					'type'    => 'select',
					'desc'    => esc_html__( 'Used for breadcrumbs, post meta categories and related items.', 'total' ),
					'choices' => __CLASS__ . '::choices_taxonomies',
				],
			]
		];
	}

	/**
	 * Post Type archive options.
	 */
	public static function type_archive_metabox() {
		return [
			'id'       => 'total_ptu_type_archive',
			'title'    => esc_html__( 'Theme Settings', 'total' ) . ' - ' . esc_html__( 'Archives', 'total' ),
			'screen'   => [ 'ptu' ],
			'context'  => 'normal',
			'priority' => 'low',
			'fields'   => [
				[
					'name' => esc_html__( 'Custom Title', 'total' ),
					'id'   => 'total_archive_page_header_title',
					'type' => 'text',
				],
				[
					'name'    => esc_html__( 'Title Style', 'total' ),
					'id'      => 'total_archive_page_header_title_style',
					'type'    => 'select',
					'choices' => 'wpex_get_page_header_styles',
				],
				[
					'name'    => esc_html__( 'Layout', 'total' ),
					'id'      => 'total_archive_layout',
					'type'    => 'select',
					'choices' => 'wpex_get_post_layouts',
				],
				[
					'name' => esc_html__( 'Post Count', 'total' ),
					'id'   => 'total_archive_posts_per_page',
					'type' => 'text',
					'desc' => esc_html__( 'How many posts do you want to display before showing the post pagination? Enter -1 to display all of them without pagination.', 'total' ),
				],
				[
					'name'    => esc_html__( 'Pagination Style', 'total' ),
					'id'      => 'total_archive_pagination_style',
					'type'    => 'select',
					'choices' => 'TotalTheme\\Pagination\\Core::choices',
				],
				[
					'name'    => esc_html__( 'Columns', 'total' ),
					'id'      => 'total_archive_grid_columns',
					'type'    => 'select',
					'choices' => __CLASS__ . '::choices_grid_columns',
				],
				[
					'name'    => esc_html__( 'Grid Style', 'total' ),
					'id'      => 'total_archive_grid_style',
					'type'    => 'select',
					'choices' => [
						''        => esc_html__( 'Default', 'total' ),
						'masonry' => esc_html__( 'Masonry', 'total' ),
					],
				],
				[
					'name'    => esc_html__( 'Gap', 'total' ),
					'id'      => 'total_archive_grid_gap',
					'type'    => 'select',
					'choices' => 'wpex_column_gaps',
				],
				[
					'name'    => esc_html__( 'Card Style', 'total' ),
					'id'      => 'total_entry_card_style',
					'type'    => 'select',
					'choices' => __CLASS__ . '::choices_card_styles',
				],
				[
					'name'    => esc_html__( 'Image Overlay', 'total' ),
					'id'      => 'total_entry_overlay_style',
					'type'    => 'select',
					'choices' => 'wpex_overlay_styles_array',
				],
				[
					'name'    => esc_html__( 'Blocks (non-card style)', 'total' ),
					'id'      => 'total_entry_blocks',
					'type'    => 'multi_select',
					'default' => [ 'media', 'title', 'meta', 'content', 'readmore' ],
					'choices' => Entry_Blocks::choices(),
				//	'choices' => 'TotalTheme\\CPT\\Entry_Blocks::choices', // @todo
				],
				[
					'name'    => esc_html__( 'Meta (non-card style)', 'total' ),
					'id'      => 'total_entry_meta_blocks',
					'type'    => 'multi_select',
					'default' => [ 'date', 'author', 'categories', 'comments' ],
					'choices' => [
						'date'       => esc_html__( 'Date', 'total' ),
						'author'     => esc_html__( 'Author', 'total' ),
						'categories' => esc_html__( 'Categories (Main Taxonomy)', 'total' ),
						'comments'   => esc_html__( 'Comments', 'total' ),
					],
				],
				[
					'name'        => esc_html__( 'Excerpt Length', 'total' ),
					'id'          => 'total_entry_excerpt_length',
					'type'        => 'number', // important to allow 0 to save and -1
					'min'         => '-1',
					'step'        => '1',
					'max'         => '9999',
					'placeholder' => '40',
					'desc'        => esc_html__( 'Number of words to display for your excerpt. Enter -1 to display the full post content. Note: custom excerpts are not trimmed.', 'total' ),
				],
				[
					'name' => esc_html__( 'Read More Button Text', 'total' ),
					'id'   => 'total_entry_readmore_text',
					'type' => 'text',
				],
			],

		];
	}

	/**
	 * Post Type single options.
	 */
	public static function type_single_metabox() {
		return [
			'id'       => 'total_ptu_type_single',
			'title'    => esc_html__( 'Theme Settings', 'total' ) . ' - ' . esc_html__( 'Single Post', 'total' ),
			'screen'   => [ 'ptu' ],
			'context'  => 'normal',
			'priority' => 'low',
			'fields'   => [
				[
					'name'    => esc_html__( 'Dynamic Template', 'total' ),
					'id'      => 'total_singular_template_id',
					'type'    => 'select',
					'desc'    => esc_html__( 'Select a template to be used for your singular post design.', 'total' ),
					'choices' => 'wpex_choices_dynamic_templates',
				],
				[
					'name' => esc_html__( 'Title', 'total' ),
					'id'   => 'total_page_header_title',
					'type' => 'text',
					'desc' => esc_html__( 'Use {{title}} to display the current title.', 'total' ),
				],
				[
					'name'    => esc_html__( 'Title Style', 'total' ),
					'id'      => 'total_page_header_title_style',
					'type'    => 'select',
					'choices' => __CLASS__ . '::choices_page_header_styles',
				],
				[
					'name'    => esc_html__( 'Title tag', 'total' ),
					'id'      => 'total_page_header_title_tag',
					'type'    => 'select',
					'choices' => [
						''     => esc_html__( 'Default', 'total' ),
						'h1'   => 'h1',
						'h2'   => 'h2',
						'h3'   => 'h3',
						'h4'   => 'h4',
						'h5'   => 'h5',
						'h6'   => 'h6',
						'div'  => 'div',
						'span' => 'span',
					],
				],
				[
					'name'    => esc_html__( 'Layout', 'total' ),
					'id'      => 'total_post_layout',
					'type'    => 'select',
					'choices' => 'wpex_get_post_layouts',
				],
				[
					'name'    => esc_html__( 'Blocks', 'total' ),
					'id'      => 'total_single_blocks',
					'type'    => 'multi_select',
					'default' => [
						'media',
						'title',
						'meta',
						'post-series',
						'content',
						'page-links',
						'share',
						'author-bio',
						'related',
						'comments'
					],
					'choices' => Single_Blocks::choices(),
				//	'choices' => 'TotalTheme\\CPT\\Single_Blocks::choices', // @todo
				],
				[
					'name'    => esc_html__( 'Meta', 'total' ),
					'id'      => 'total_single_meta_blocks',
					'type'    => 'multi_select',
					'default' => [ 'date', 'author', 'categories', 'comments' ],
					'choices' => [
						'date'       => esc_html__( 'Date', 'total' ),
						'author'     => esc_html__( 'Author', 'total' ),
						'categories' => esc_html__( 'Categories (Main Taxonomy)', 'total' ),
						'comments'   => esc_html__( 'Comments', 'total' ),
					],
				],
				[
					'name'    => esc_html__( 'Display Next/Previous Links?', 'total' ),
					'id'      => 'total_next_prev',
					'type'    => 'checkbox',
					'default' => true,
				],
			]

		];
	}

	/**
	 * Post Type related options.
	 */
	public static function type_related_metabox() {
		return [
			'id'       => 'total_ptu_related',
			'title'    => esc_html__( 'Theme Settings', 'total' ) . ' - ' . esc_html__( 'Related Posts', 'total' ),
			'screen'   => [ 'ptu' ],
			'context'  => 'normal',
			'priority' => 'low',
			'fields'   => [
				[
					'name'    => esc_html__( 'Related By', 'total' ),
					'id'      => 'total_related_taxonomy',
					'type'    => 'select',
					'choices' => __CLASS__ . '::choices_related_by',
				],
				[
					'name'    => esc_html__( 'Order', 'total' ),
					'id'      => 'total_related_order',
					'type'    => 'select',
					'choices' => [
						''     => esc_html__( 'Default', 'total' ),
						'desc' => esc_html__( 'DESC', 'total' ),
						'asc'  => esc_html__( 'ASC', 'total' ),
					],
				],
				[
					'name'    => esc_html__( 'Order By', 'total' ),
					'id'      => 'total_related_orderby',
					'type'    => 'select',
					'choices' => [
						''     => esc_html__( 'Default', 'total' ),
						'date'          => esc_html__( 'Date', 'total' ),
						'title'         => esc_html__( 'Title', 'total' ),
						'modified'      => esc_html__( 'Modified', 'total' ),
						'author'        => esc_html__( 'Author', 'total' ),
						'rand'          => esc_html__( 'Random', 'total' ),
						'comment_count' => esc_html__( 'Comment Count', 'total' ),
					],
				],
				[
					'name' => esc_html__( 'Post Count', 'total' ),
					'id'   => 'total_related_count',
					'type' => 'text',
				],
				[
					'name'    => esc_html__( 'Columns', 'total' ),
					'id'      => 'total_related_columns',
					'type'    => 'select',
					'choices' => __CLASS__ . '::choices_grid_columns',
				],
				[
					'name'    => esc_html__( 'Gap', 'total' ),
					'id'      => 'total_related_gap',
					'type'    => 'select',
					'choices' => 'wpex_column_gaps',
				],
				[
					'name'    => esc_html__( 'Card Style', 'total' ),
					'id'      => 'total_related_entry_card_style',
					'type'    => 'select',
					'choices' => __CLASS__ . '::choices_card_styles',
				],
				[
					'name'    => esc_html__( 'Image Overlay', 'total' ),
					'id'      => 'total_related_entry_overlay_style',
					'type'    => 'select',
					'choices' => 'wpex_overlay_styles_array',
				],
				[
					'name'        => esc_html__( 'Excerpt Length', 'total' ),
					'id'          => 'total_related_entry_excerpt_length',
					'type'        => 'number', // important to allow 0 to save and -1
					'min'         => '-1',
					'step'        => '1',
					'max'         => '9999',
					'placeholder' => '15',
					'desc'        => esc_html__( 'Number of words to display for your excerpt. Enter -1 to display the full post content. Note: custom excerpts are not trimmed.', 'total' ),
				],
			]
		];
	}

	/**
	 * Taxonomy options.
	 */
	public static function tax_metabox() {
		return [
			'id'       => 'total_ptu_tax',
			'title'    => esc_html__( 'Theme Settings', 'total' ),
			'screen'   => [ 'ptu_tax' ],
			'context'  => 'normal',
			'priority' => 'low',
			'fields'   => [
				[
					'name'    => esc_html__( 'Main Page', 'total' ),
					'id'      => 'total_tax_main_page',
					'type'    => 'select',
					'desc'    => esc_html__( 'Used for breadcrumbs.', 'total' ),
					'choices' => __CLASS__ . '::choices_tax_main_page',
				],
				[
					'name'    => esc_html__( 'Title Style', 'total' ),
					'id'      => 'total_tax_page_header_title_style',
					'type'    => 'select',
					'choices' => __CLASS__ . '::choices_page_header_styles',
				],
				[
					'name' => esc_html__( 'Custom Title', 'total' ),
					'id'   => 'total_tax_page_header_title',
					'type' => 'text',
					'desc' => esc_html__( 'Use {{title}} to display the current title.', 'total' ),
				],
				[
					'name'    => esc_html__( 'Template', 'total' ),
					'id'      => 'total_tax_template_id',
					'type'    => 'select',
					'choices' => 'wpex_choices_dynamic_templates',
				],
				[
					'name'    => esc_html__( 'Layout', 'total' ),
					'id'      => 'total_tax_layout',
					'type'    => 'select',
					'choices' => 'wpex_get_post_layouts',
				],
				[
					'name' => esc_html__( 'Post Count', 'total' ),
					'id'   => 'total_tax_posts_per_page',
					'type' => 'text',
					'desc' => esc_html__( 'How many posts do you want to display before showing the post pagination? Enter -1 to display all of them without pagination.', 'total' ),
				],
				[
					'name'    => esc_html__( 'Pagination Style', 'total' ),
					'id'      => 'total_tax_pagination_style',
					'type'    => 'select',
					'choices' => 'TotalTheme\\Pagination\\Core::choices',
				],
				[
					'name'    => esc_html__( 'Sidebar', 'total' ),
					'id'      => 'total_tax_sidebar',
					'type'    => 'select',
					'choices' => 'wpex_choices_widget_areas',
				],
				[
					'name'    => esc_html__( 'Columns', 'total' ),
					'id'      => 'total_tax_grid_columns',
					'type'    => 'select',
					'choices' => 'wpex_grid_columns',
				],
				[
					'name'    => esc_html__( 'Grid Style', 'total' ),
					'id'      => 'total_tax_grid_style',
					'type'    => 'select',
					'choices' => [
						''        => esc_html__( 'Default', 'total' ),
						'masonry' => esc_html__( 'Masonry', 'total' ),
					],
				],
				[
					'name'    => esc_html__( 'Gap', 'total' ),
					'id'      => 'total_tax_grid_gap',
					'type'    => 'select',
					'choices' => 'wpex_column_gaps',
				],
				[
					'name'    => esc_html__( 'Description Position', 'total' ),
					'id'      => 'total_tax_term_description_position',
					'type'    => 'select',
					'choices' => [
						'subheading' => esc_html__( 'As Subheading', 'total' ),
						'above_loop' => esc_html__( 'Before Your Posts', 'total' ),
					],
				],
				[
					'name'    => esc_html__( 'Page Header Thumbnail', 'total' ),
					'id'      => 'total_tax_term_page_header_image_enabled',
					'type'    => 'checkbox',
					'default' => true,
				],
				[
					'name'    => esc_html__( 'Card Style', 'total' ),
					'id'      => 'total_tax_entry_card_style',
					'type'    => 'select',
					'choices' => __CLASS__ . '::choices_card_styles',
				],
				[
					'name'    => esc_html__( 'Image Overlay', 'total' ),
					'id'      => 'total_tax_entry_overlay_style',
					'type'    => 'select',
					'choices' => 'wpex_overlay_styles_array',
				],
				[
					'name'    => esc_html__( 'Image Size', 'total' ),
					'id'      => 'total_tax_entry_image_size',
					'type'    => 'select',
					'choices' => __CLASS__ . '::choices_image_size',
				],
				[
					'name' => esc_html__( 'Excerpt Length', 'total' ),
					'id'   => 'total_tax_entry_excerpt_length',
					'type' => 'number', // important to allow 0 to save and -1
					'min'  => '-1',
					'step' => '1',
					'max'  => '9999',
					'desc' => esc_html__( 'Number of words to display for your excerpt. Enter -1 to display the full post content. Note: custom excerpts are not trimmed.', 'total' ),
				],
			]
		];
	}

	/**
	 * Grid column choices.
	 */
	public static function choices_grid_columns() {
		return [ '' => esc_html__( 'Default', 'total' ) ] + wpex_grid_columns();
	}

	/**
	 * Card style choices.
	 */
	public static function choices_card_styles() {
		$choices = wpex_choices_card_styles();
		unset( $choices['woocommerce'] );
		return $choices;
	}

	/**
	 * Related by choices.
	 */
	public static function choices_related_by() {
		return array_merge( self::choices_taxonomies(), [ 'null' => esc_html__( 'Anything', 'total' ) ] );
	}

	/**
	 * Image size selector.
	 */
	public static function choices_image_size() {
		$choices = [
			'' => esc_html__( 'Default', 'total' ),
		];
		$get_sizes = wpex_get_thumbnail_sizes();
		if ( $get_sizes ) {
			foreach ( $get_sizes as $size => $dims ) {
				$choices[$size] = $size;
			}
		}
		return $choices;
	}

	/**
	 * Page choices for select field.
	 */
	public static function choices_pages() {
		$choices = [
			'' => esc_html__( 'None', 'total' ),
		];
		$get_pages = get_pages();
		if ( $get_pages && ! is_wp_error( $get_pages ) ) {
			foreach ( $get_pages as $page ) {
				$choices[$page->ID] = $page->post_title;
			}
		}
		return $choices;
	}

	/**
	 * Page choices for the taxonomy main page select.
	 */
	public static function choices_tax_main_page() {
		$choices = [
			'' => esc_html__( 'None', 'total' ),
		];
		$get_pages = get_pages();
		if ( $get_pages && ! is_wp_error( $get_pages ) ) {
			foreach ( $get_pages as $page ) {
				$choices[$page->ID] = $page->post_title;
			}
		}
		$types = self::get_post_types();
		if ( is_array( $types ) ) {
			foreach ( $types as $type_name => $ptu_id ) {
				$post_type = get_post_type_object( $type_name );
				if ( is_object( $post_type ) && ! is_wp_error( $post_type ) ) {
					$choices['pt_archive_' . $type_name] = sprintf( esc_html__( 'Post Type Archive: %s', 'total' ), $post_type->labels->name );
				}
			}
		}
		return $choices;
	}

	/**
	 * Taxonomy select.
	 */
	public static function choices_taxonomies() {
		$choices = [
			'' => esc_html__( 'Select', 'total' ),
		];
		$taxonomies = get_taxonomies( [
			'public' => true,
		], 'objects' );
		if ( $taxonomies && ! is_wp_error( $taxonomies ) ) {
			foreach ( $taxonomies as $taxonomy ) {
				$choices[$taxonomy->name] = esc_html( $taxonomy->label ) . ' (' . $taxonomy->name . ')';
			}
		}
		return $choices;
	}

	/**
	 * Page Header Styles Select.
	 */
	public static function choices_page_header_styles() {
		$styles = [
			''                 => esc_html__( 'Default','total' ),
			'default'          => esc_html__( 'Standard', 'total' ),
			'centered'         => esc_html__( 'Centered', 'total' ),
			'centered-minimal' => esc_html__( 'Centered Minimal', 'total' ),
			'background-image' => esc_html__( 'Background Image', 'total' ),
			'hidden'           => esc_html__( 'Hidden (Disabled)', 'total' ),
		];
		$styles = (array) apply_filters( 'wpex_page_header_styles', $styles );
		return $styles;
	}

	/**
	 * Get post types and store in class variable.
	 */
	public static function get_post_types() {
		if ( is_callable( 'PTU\\PostTypes::get_registered_items' ) ) {
			return \PTU\PostTypes::get_registered_items(); // Added in PTU 1.1
		}

		// @todo deprecate and force people to update PTU.
		$types = [];
		$get_types = get_posts( [
			'numberposts' 	   => -1,
			'post_type' 	   => 'ptu',
			'post_status'      => 'publish',
			'suppress_filters' => false, // allow caching & translating.
			'fields'           => 'ids',
		] );
		if ( $get_types ) {
			foreach ( $get_types as $id ) {
				$name = get_post_meta( $id, '_ptu_name', true );
				if ( $name ) {
					$types[ $name ] = $id;
				}
			}
		}
		return $types;
	}

	/**
	 * Get taxonomies and store in class variable.
	 */
	public static function get_taxonomies() {
		if ( is_callable( 'PTU\\Taxonomies::get_registered_items' ) ) {
			return \PTU\Taxonomies::get_registered_items(); // Added in PTU 1.1
		}

		// @todo deprecate and force people to update PTU.
		$taxonomies = [];
		$get_taxes = get_posts( [
			'numberposts' 	   => -1,
			'post_type' 	   => 'ptu_tax',
			'post_status'      => 'publish',
			'suppress_filters' => false, // allow caching & translating.
			'fields'           => 'ids',
		] );
		if ( $get_taxes ) {
			foreach ( $get_taxes as $id ) {
				$name = get_post_meta( $id, '_ptu_name', true );
				if ( $name ) {
					$taxonomies[ $name ] = $id;
				}
			}
		}
		return $taxonomies;
	}

	/**
	 * Return post typemeta value.
	 */
	public static function get_setting_value( $post_type, $setting_id ) {
		$types = self::get_post_types();
		if ( ! empty( $types[$post_type] ) ) {
			return get_post_meta( $types[$post_type], $setting_id, true );
		}
	}

	/**
	 * Return meta value.
	 */
	public static function get_tax_setting_value( $tax, $setting_id ) {
		$taxes = self::get_taxonomies();
		if ( ! empty( $taxes[$tax] ) ) {
			return get_post_meta( $taxes[$tax], $setting_id, true );
		}
	}

	/**
	 * Enable metabox for types.
	 */
	public static function metabox_main( $types ) {
		$get_types = self::get_post_types();
		if ( $get_types ) {
			foreach ( $get_types as $type => $id ) {
				if ( get_post_meta( $id, '_ptu_total_ps_meta', true ) ) {
					$types[$type] = $type;
				}
			}
		}
		return $types;
	}

	/**
	 * Enable card metabox for types.
	 */
	public static function metabox_card( $types ) {
		$get_types = self::get_post_types();
		if ( $get_types ) {
			foreach ( $get_types as $type => $id ) {
				if ( get_post_meta( $id, '_ptu_total_ps_meta_card', true ) ) {
					$types[$type] = $type;
				}
			}
		}
		return $types;
	}

	/**
	 * Enable the metabox media tab for post types.
	 */
	public static function metabox_media( $settings ) {
		if ( isset( $settings['media'] ) ) {
			$get_types = self::get_post_types();
			if ( $get_types ) {
				foreach ( $get_types as $type => $id ) {
					if ( get_post_meta( $id, '_ptu_total_ps_meta_media', true ) ) {
						$settings['media']['post_type'][] = $type;
					}
				}
			}
		}
		return $settings;
	}

	/**
	 * Enable image sizes.
	 */
	public static function wpex_image_sizes_tabs( $tabs ) {
		$types = self::get_post_types();
		if ( $types ) {
			foreach ( $types as $type => $id ) {
				if ( get_post_meta( $id, '_ptu_total_image_sizes', true ) ) {
					$postType = get_post_type_object( $type );
					if ( $postType ) {
						$tabs[$type] = $postType->labels->singular_name;
					}
				}
			}
		}
		return $tabs;
	}

	/**
	 * Add image size options.
	 */
	public static function wpex_image_sizes( $sizes ) {
		$types = self::get_post_types();
		if ( $types ) {
			foreach ( $types as $type => $id ) {
				if ( get_post_meta( $id, '_ptu_total_image_sizes', true ) ) {
					$sizes[ $type . '_archive' ] = [
						'label'   => esc_html__( 'Archive', 'total' ),
						'width'   => $type . '_archive_image_width',
						'height'  => $type . '_archive_image_height',
						'crop'    => $type . '_archive_image_crop',
						'section' => $type,
					];
					$sizes[ $type . '_single' ] = [
						'label'   => esc_html__( 'Post', 'total' ),
						'width'   => $type . '_post_image_width',
						'height'  => $type . '_post_image_height',
						'crop'    => $type . '_post_image_crop',
						'section' => $type,
					];
					$sizes[ $type . '_single_related' ] = [
						'label'   => esc_html__( 'Post Related Items', 'total' ),
						'width'   => $type . '_single_related_image_width',
						'height'  => $type . '_single_related_image_height',
						'crop'    => $type . '_single_related_image_crop',
						'section' => $type,
					];
				}
			}
		}
		return $sizes;
	}

	/**
	 * Register sidebars.
	 */
	public static function wpex_register_sidebars_array( $sidebars ) {
		$types = self::get_post_types();
		if ( $types ) {
			foreach ( $types as $type => $id ) {
				$sidebar = get_post_meta( $id, '_ptu_total_custom_sidebar', true );
				// @update to use sanitize_key.
				if ( $sidebar ) {
					$id = wp_strip_all_tags( $sidebar );
					$id = str_replace( ' ', '_', $sidebar );
					$id = strtolower( $sidebar );
					$sidebars[$id] = $sidebar;
				}
			}
		}
		return $sidebars;
	}

	/**
	 * Register post series for selected post types..4
	 */
	public static function register_post_series() {
		$types = self::get_post_types();
		if ( $types ) {
			foreach ( $types as $type => $id ) {
				$check = get_post_meta( $id, '_ptu_total_post_series', true );
				if ( wp_validate_boolean( $check ) ) {
					register_taxonomy_for_object_type( 'post_series', $type );
				}
			}
		}
	}

	/**
	 * Enable gallery metabox.
	 */
	public static function wpex_gallery_metabox_post_types( $types ) {
		$get_types = self::get_post_types();
		if ( $get_types ) {
			foreach ( $get_types as $type => $id ) {
				if ( get_post_meta( $id, '_ptu_total_post_gallery', true ) ) {
					$types[$id] = $type;
				}
			}
		}
		return $types;
	}

	/**
	 * Enable admin thumbnails.
	 */
	public static function wpex_dashboard_thumbnails_post_types( $types ) {
		$get_types = self::get_post_types();
		if ( $get_types ) {
			foreach ( $get_types as $type => $id ) {
				if ( get_post_meta( $id, '_ptu_total_show_admin_thumbnails', true ) ) {
					$types[$id] = $type;
				}
			}
		}
		return $types;
	}

}