<?php
namespace TotalTheme\Header\Menu;

use TotalTheme\Header\Core as Header;
use TotalTheme\Header\Overlay as Overlay_Header;

defined( 'ABSPATH' ) || exit;

/**
 * Header Menu.
 *
 * @package TotalTheme
 * @subpackage Header
 * @version 5.6
 */
class Search {

	/**
	 * Check if enabled or not.
	 */
	protected static $is_enabled;

	/**
	 * The style.
	 */
	protected static $style;

	/**
	 * Checks if the header menu is currently supported.
	 */
	public static function is_supported() {
		if ( ! Header::is_enabled() ) {
			return;
		}

		$check = true;

		/**
		 * Filters whether the header menu supports a search form or not.
		 *
		 * @param bool $check
		 */
		$check = (bool) apply_filters( 'totaltheme/header/menu/search/is_supported', $check );

		/*** deprecated ***/
		$check = (bool) apply_filters( 'wpex_has_menu_search', $check );

		return $check;
	}

	/**
	 * Returns header menu search style.
	 */
	public static function is_enabled() {
		$check = get_theme_mod( 'menu_search_enable', true );
		$style = self::style();

		if ( wp_validate_boolean( $check ) && $style && 'disabled' !== $style ) {
			$check = true;
		}

		/**
		 * Filters whether the header menu search is enabled.
		 *
		 * @param bool $check
		 */
		$check = (bool) apply_filters( 'totaltheme/header/menu/search/is_enabled', $check );

		return $check;
	}

	/**
	 * Returns header menu search style.
	 */
	public static function style() {
		if ( ! is_null( self::$style ) ) {
			return self::$style;
		}

		if ( ! wpex_header_menu_supports_search() ) {
			return false;
		}

		// Get style as set in the customizer.
		$style = get_theme_mod( 'menu_search_style', 'drop_down' );

		// Convert old disabled theme_mod.
		if ( 'disabled' === $style ) {
			remove_theme_mod( 'menu_search_style' );
			set_theme_mod( 'menu_search_enable', false );
		}

		// Don't allow header replace on overlay header.
		// @todo we should allow this by having the searchbar take up the full width and having a background color.
		if ( 'header_replace' === $style && Overlay_Header::is_enabled() ) {
			$style = 'overlay';
		}

		/**
		 * Filters the header menu search style
		 *
		 * @param string $style
		 */
		$style = (string) apply_filters( 'totaltheme/header/menu/search/style', $style );

		/*** deprecated ***/
		$style = (string) apply_filters( 'wpex_menu_search_style', $style );

		if ( ! $style ) {
			$style = 'drop_down'; // style must never be empty.
		}

		self::$style = $style;

		return self::$style;
	}

	/**
	 * Returns header menu search form placeholder text.
	 */
	public static function get_placeholder_text() {
		$placeholder = wpex_get_translated_theme_mod( 'menu_search_placeholder' );

		if ( ! $placeholder ) {
			$placeholder = esc_html__( 'Search', 'total' );
			$style = self::style();
			if ( 'overlay' === $style || 'header_replace' === $style ) {
				$placeholder = esc_html__( 'Type then hit enter to search&hellip;', 'total' );
			}
		}

		/**
		 * Filters the header menu searchform placeholder text.
		 *
		 * @param string $placeholder
		 */
		$placeholder = (string) apply_filters( 'totaltheme/header/menu/search/placeholder', $placeholder );

		/*** deprecated ***/
		$placeholder = (string) apply_filters( 'wpex_get_header_menu_search_form_placeholder', $placeholder );

		return $placeholder;
	}

	/**
	 * Renders the dropdown widget classes.
	 */
	public static function drop_widget_class() {
		$class = [
			'header-searchform-wrap',
		];

		$widget_class = wpex_get_header_drop_widget_class();

		if ( $widget_class ) {
			$class = array_merge( $class, $widget_class );
		}

		$class[] = 'wpex-p-15';

		if ( $class ) {
			echo 'class="' . esc_attr( implode( ' ', $class ) ) . '"';
		}
	}

	/**
	 * Returns header menu search form.
	 */
	public static function get_form() {
		if ( WPEX_WOOCOMMERCE_ACTIVE && get_theme_mod( 'woo_header_product_searchform', false ) ) {
			$form = get_product_search_form( false );
		} else {
			$form = get_search_form( false );
		}

		/**
		 * Filters the header menus search form html.
		 *
		 * @param string $form
		 */
		$form = (string) apply_filters( 'totaltheme/header/menu/search/form', $form );

		/*** deprecated ***/
		$form = (string) apply_filters( 'wpex_get_header_menu_search_form', $form );

		return $form;
	}

	/**
	 * Returns header menu search icon choices.
	 */
	public static function icon_choices() {
		$choices = [
			'search',
			'material/search',
			'ionicons/search',
			'ionicons/search-outline',
			'ionicons/search-sharp',
		];

		/**
		 * Filters the header menu search icon choices.
		 *
		 * @param array $choices
		 */
		$choices = (array) apply_filters( 'totaltheme/header/menu/search/icon_choices', $choices );

		/*** deprecated **/
		$choices = apply_filters( 'wpex_header_menu_icon_choices', $choices );

		return $choices;
	}

	/**
	 * Returns header menu search icon html.
	 */
	protected static function get_icon_html() {
		$icon = get_theme_mod( 'menu_search_icon' ) ?: 'search';
		return wpex_get_theme_icon_html( $icon, 'wpex-menu-search-icon' );
	}

	/**
	 * Inserts the header menu search icon into the menu.
	 */
	public static function insert_icon( $items, $args ) {
		if ( Header::has_flex_container() || ! self::is_enabled() ) {
			return $items;
		}

		$aria_controls = '';

		/*** deprecated ***/
		$search_icon_theme_locations = (array) apply_filters( 'wpex_menu_search_icon_theme_locations', [ 'main_menu' ] );

		// Check menu location.
		if ( ! in_array( $args->theme_location, $search_icon_theme_locations ) ) {
			return $items;
		}

		// Get search style.
		$search_style = self::style();

		// Get header style.
		$header_style = Header::style();

		// Define classes.
		$li_classes = 'search-toggle-li menu-item wpex-menu-extra';
		$a_classes  = 'site-search-toggle';

		// Remove icon margin.
		if ( 'six' !== $header_style ) {
			$li_classes .= ' no-icon-margin';
		}

		// Define vars based on search style.
		switch ( $search_style ) {
			case 'overlay':
				$a_classes .= ' search-overlay-toggle';
				$aria_controls = 'wpex-searchform-overlay';
				break;
			case 'drop_down':
				$a_classes .= ' search-dropdown-toggle';
				$aria_controls = 'searchform-dropdown';
				break;
			case 'header_replace':
				$a_classes .= ' search-header-replace-toggle';
				$aria_controls = 'searchform-header-replace';
				break;
		}

		// Ubermenu integration.
		if ( class_exists( 'UberMenu' ) && apply_filters( 'wpex_add_search_toggle_ubermenu_classes', true ) ) {
			$li_classes .= ' ubermenu-item-level-0 ubermenu-item';
			$a_classes  .= ' ubermenu-target ubermenu-item-layout-default ubermenu-item-layout-text_only';
		}

		// Max Mega Menu integration.
		if ( function_exists( 'max_mega_menu_is_enabled' ) && max_mega_menu_is_enabled( $args->theme_location ) ) {
			$li_classes .= ' mega-menu-item';
			$a_classes  .= ' mega-menu-link';
		}

		// Add search icon and dropdown style.
		$menu_search = '';
		$menu_search .= '<li class="' . esc_attr( $li_classes ) . '">';

			$a_attributes = [
				'href'          => '#',
				'class'         => esc_attr( $a_classes ),
				'role'          => 'button',
				'aria-expanded' => 'false',
				'aria-controls' => $aria_controls,
			];

			$a_aria = wpex_get_aria_label( 'search' );
			if ( $a_aria ) {
				$a_attributes['aria-label'] = esc_attr( $a_aria );
			}

			$menu_search .= '<a ' . wpex_parse_attrs( $a_attributes ) . '>';

				$menu_search .= '<span class="link-inner">';

					$text = esc_html__( 'Search', 'total' );

					/**
					 * Filters the header search icon text.
					 *
					 * @param string $text
					 */
					$text = (string) apply_filters( 'totaltheme/header/menu/search/icon_label', $text );

					/*** deprecated ***/
					$text = (string) apply_filters( 'wpex_header_search_text', $text );

					$icon = self::get_icon_html();

					if ( 'six' === $header_style ) {
						$menu_search .= $icon . '<span class="wpex-menu-search-text">' . esc_html( $text ) . '</span>';
					} else {
						$menu_search .= '<span class="wpex-menu-search-text">' . esc_html( $text ) . '</span>' . $icon;
					}

					$menu_search .= '</span>';

			$menu_search .= '</a>';

			if ( 'drop_down' === $search_style && true === wpex_maybe_add_header_drop_widget_inline( 'search' ) ) {
				ob_start();
				wpex_get_template_part( 'header_search_dropdown' );
				$menu_search .= ob_get_clean();
			}

		$menu_search .= '</li>';

		/**
		 * Filters the header menu search icon position.
		 *
		 * @param $position | options: start or end.
		 */
		$menu_search_position = apply_filters( 'wpex_header_menu_search_position', 'end' );

		switch ( $menu_search_position ) {
			case 'start':
				$items = $menu_search . $items;
				break;
			case 'end':
			default;
				$items = $items . $menu_search;
				break;
		}

		return $items;
	}

}