<?php
defined( 'ABSPATH' ) || exit;

/*-------------------------------------------------------------------------------*/
/* [ Table of contents ]
/*-------------------------------------------------------------------------------*/

	# General
	# Archives
	# Entries
	# Single
	# Related
	# Social
	# Cards
	# Deprecated

/*-------------------------------------------------------------------------------*/
/* [ General ]
/*-------------------------------------------------------------------------------*/

/**
 * Check if a staff member has a defined position.
 */
function wpex_has_staff_member_position( $post = '' ) {
	return (bool) wpex_get_staff_member_position( $post );
}

/**
 * Returns staff members position.
 */
function wpex_get_staff_member_position( $post = '' ) {
	$post = get_post( $post );

	if ( ! $post ) {
		return;
	}

	$position = get_post_meta( get_the_ID(), 'wpex_staff_position', true );

	/**
	 * Filters the staff member position.
	 *
	 * @param string $position
	 * @param object $post
	 */
	$position = (string) apply_filters( 'wpex_staff_member_position', $position, $post );

	return $position;
}

/*-------------------------------------------------------------------------------*/
/* [ Archives ]
/*-------------------------------------------------------------------------------*/

/**
 * Returns staff archive grid style.
 */
function wpex_staff_archive_grid_style() {
	$style = get_theme_mod( 'staff_archive_grid_style', 'fit-rows' );

	if ( ! $style ) {
		$style = 'fit-rows';
	}

	/**
	 * Filters the staff grid style.
	 *
	 * @param string $style.
	 */
	$style = (string) apply_filters( 'wpex_staff_archive_grid_style', $style );

	return $style;
}

/**
 * Returns the staff loop top class.
 */
function wpex_staff_loop_top_class() {
	$class = wpex_get_staff_wrap_classes();

	/**
	 * Filters the staff loop top element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_loop_top_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', $class ) ) . '"';
	}
}

/**
 * Returns correct classes for the staff wrap.
 */
function wpex_get_staff_wrap_classes() {
	$classes = [
		'wpex-row',
	];

	$grid_style = wpex_staff_archive_grid_style();

	// Masonry class.
	if ( 'masonry' === $grid_style || 'no-margins' === $grid_style ) {
		$classes[] = 'wpex-masonry-grid';
		wpex_enqueue_isotope_scripts(); // dynamically load masonry scripts.
	}

	// Add grid style.
	$classes[] = 'staff-' . sanitize_html_class( $grid_style );

	// Add gap class.
	if ( 'no-margins' === $grid_style ) {
		$classes[] = wpex_gap_class( '0px' );
	} elseif ( $gap = get_theme_mod( 'staff_archive_grid_gap' ) ) {
		$classes[] = wpex_gap_class( $gap );
	}

	$classes[] = 'wpex-clr';

	$classes = array_map( 'esc_attr', $classes );

	/**
	 * Filters the staff wrap element class.
	 *
	 * @param array $classes
	 */
	$classes = (array) apply_filters( 'wpex_staff_wrap_classes', $classes );

	return implode( ' ', $classes );

}

/**
 * Returns staff archive columns.
 */
function wpex_staff_archive_columns() {
	$columns = get_theme_mod( 'staff_entry_columns' );

	if ( ! $columns ) {
		$columns = 3;
	}

	/**
	 * Filters the number of columns to display for the staff entries.
	 *
	 * @param int|string $columns
	 */
	$columns = apply_filters( 'wpex_staff_archive_columns', $columns );

	return $columns;
}

/**
 * Returns correct classes for the staff grid.
 */
if ( ! function_exists( 'wpex_staff_column_class' ) ) {
	function wpex_staff_column_class( $query ) {
		if ( 'related' === $query ) {
			return wpex_row_column_width_class( get_theme_mod( 'staff_related_columns', '3' ) );
		} else {
			return wpex_row_column_width_class( get_theme_mod( 'staff_entry_columns', '3' ) );
		}
	}
}

/**
 * Checks if match heights are enabled for the staff.
 */
if ( ! function_exists( 'wpex_staff_match_height' ) ) {
	function wpex_staff_match_height() {
		if ( ! get_theme_mod( 'staff_archive_grid_equal_heights' ) ) {
			return false;
		}

		$grid_style = wpex_staff_archive_grid_style();

		if ( 'fit-rows' !== $grid_style ) {
			return false;
		}

		$columns = absint( wpex_get_array_first_value( wpex_staff_archive_columns() ) );

		if ( $columns && $columns > 1 ) {
			return true;
		}

	}
}

/*-------------------------------------------------------------------------------*/
/* [ Entries ]
/*-------------------------------------------------------------------------------*/

/**
 * Get staff entry supported media types.
 */
function wpex_staff_entry_supported_media() {
	$supported_media = array(
		'thumbnail',
	);

	/**
	 * Filters the staff entry supported media types.
	 *
	 * @param array $supported_media
	 */
	$supported_media = (array) apply_filters( 'wpex_staff_entry_supported_media', $supported_media );

	return $supported_media;
}

/**
 * Get Staff entry media type.
 */
function wpex_staff_entry_media_type() {
	$supported_media = wpex_staff_entry_supported_media();

	if ( in_array( 'thumbnail', $supported_media ) && has_post_thumbnail() ) {
		$type = 'thumbnail';
	} else {
		$type = '';
	}

	/**
	 * Filters the staff entry media type.
	 *
	 * @param string $typ
	 */
	$type = (string) apply_filters( 'wpex_staff_entry_media_type', $type );

	return $type;
}

/**
 * Staff entry class.
 */
function wpex_staff_entry_class( $loop = '' ) {
	if ( ! $loop ) {
		$loop = wpex_get_loop_instance();
	}

	// Default entry classes.
	$class = array(
		'staff-entry',
	);

	// Add grid column class.
	if ( $col_class = wpex_staff_column_class( $loop ) ) {
		$class[] = 'col';
		$class[] = $col_class;
	}

	// Add loop counter class.
	if ( $loop_counter = wpex_get_loop_counter() ) {
		$class[] = 'col-' . absint( $loop_counter );
	}

	// Masonry Classes.
	if ( 'archive' === $loop ) {

		$grid_style = wpex_staff_archive_grid_style();

		if ( 'masonry' === $grid_style || 'no-margins' === $grid_style ) {
			$class[] = 'wpex-masonry-col';
		}

	}

	/**
	 * Filters the staff entry class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_entry_class', $class );

	post_class( $class );
}


/**
 * Staff entry inner class.
 */
function wpex_staff_entry_inner_class( $loop = '' ) {
	if ( ! $loop ) {
		$loop = wpex_get_loop_instance();
	}

	$class = array(
		'staff-entry-inner',
		'wpex-last-mb-0',
		'wpex-clr',
	);

	/**
	 * Filters the staff entry inner class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_entry_inner_class', $class, $loop );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Staff entry media class.
 */
function wpex_staff_entry_media_class() {
	$media_type = wpex_staff_entry_media_type();

	$class = array(
		'staff-entry-media',
		'entry-media',
		'wpex-mb-20'
	);

	if ( 'thumbnail' === $media_type ) {

		if ( $overlay = wpex_overlay_classes() ) {
			$class[] = $overlay;
		}

		if ( $animation = wpex_get_entry_image_animation_classes() ) {
			$class[] = $animation;
		}

	}

	/**
	 * Filters the staff entry media class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_entry_media_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Staff entry title class.
 */
function wpex_staff_entry_title_class() {
	$class = array(
		'staff-entry-title',
		'entry-title',
		'wpex-mb-5',
	);

	/**
	 * Filters the staff entry title class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_entry_title_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}


/**
 * Staff entry content class.
 */
function wpex_staff_entry_content_class() {
	$class = array(
		'staff-entry-details',
	);

	if ( wpex_staff_match_height() ) {
		$class[] = 'match-height-content';
	}

	$class[] = 'wpex-last-mb-0';
	$class[] = 'wpex-clr';

	/**
	 * Filters the staff entry content element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_entry_content_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Staff entry excerpt class.
 */
function wpex_staff_entry_excerpt_class() {
	$class = array(
		'staff-entry-excerpt',
		'wpex-my-15',
		'wpex-last-mb-0',
	);

	/**
	 * Filters the staff entry excerpt element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_entry_excerpt_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Staff entry position class.
 */
function wpex_staff_entry_position_class() {
	$class = array(
		'staff-entry-position',
		'entry-position',
		'wpex-mb-15',
		'wpex-text-sm',
		//'wpex-text-gray-600',
		'wpex-text-3',
	);

	/**
	 * Filters the staff entry position element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_entry_position_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Echo staff entry thumbnail.
 */
function wpex_staff_entry_thumbnail() {
	echo wpex_get_staff_entry_thumbnail();
}

/**
 * Returns correct thumbnail HTML for the staff entries.
 *
 * @since 2.0.0
 */
function wpex_get_staff_entry_thumbnail( $loop = '', $args = array() ) {
	if ( ! $loop ) {
		$loop = wpex_get_loop_instance();
	}

	$defaults = array(
		'size'  => ( 'archive' === $loop ) ? 'staff_entry' : 'staff_related',
		'class' => 'staff-entry-media-img staff-entry-img wpex-align-middle',
	);

	$args = wp_parse_args( $args, $defaults );

	/**
	 * Filters the staff entry thumbnail args.
	 *
	 * @param array $args
	 */
	$args = (array) apply_filters( 'wpex_get_staff_entry_thumbnail_args', $args );

	return wpex_get_post_thumbnail( $args );
}

/**
 * Check if staff entry content is enabled.
 */
function wpex_has_staff_entry_content() {
	$loop = wpex_get_loop_instance();

	if ( 'related' === $loop ) {
		$check = get_theme_mod( 'staff_related_excerpts', true );
	} else {
		$check = get_theme_mod( 'staff_entry_details', true );
	}

	/**
	 * Filters whether the staff entry should display an excerpt.
	 *
	 * @param bool $check
	 */
	$check = (bool) apply_filters( 'wpex_has_staff_entry_content', $check );

	return $check;

}

/**
 * Returns staff entry excerpt length.
 */
function wpex_staff_entry_excerpt_length() {
	$length = get_theme_mod( 'staff_entry_excerpt_length', 20 );

	if ( 'related' === wpex_get_loop_instance() ) {
		$length = get_theme_mod( 'staff_related_entry_excerpt_length', $length );
	}

	/**
	 * Filters the staff entry excerpt length.
	 *
	 * @param int $length
	 */
	$length = (int) apply_filters( 'wpex_staff_entry_excerpt_length', $length );

	return $length;
}

/*-------------------------------------------------------------------------------*/
/* [ Single ]
/*-------------------------------------------------------------------------------*/

/**
 * Get staff single supported media types.
 */
function wpex_staff_single_supported_media() {
	$supported_media = array(
		'gallery',
		'thumbnail',
	);

	/**
	 * Filters the staff single post supported media types.
	 *
	 * @param array $supported_media
	 */
	$supported_media = (array) apply_filters( 'wpex_staff_single_supported_media', $supported_media );

	return $supported_media;
}

/**
 * Get staff single format.
 */
function wpex_staff_single_media_type() {
	$supported_media = wpex_staff_single_supported_media();

	if ( in_array( 'gallery', $supported_media ) && wpex_has_post_gallery() ) {
		$type = 'gallery';
	} elseif ( in_array( 'thumbnail', $supported_media ) && has_post_thumbnail() ) {
		$type = 'thumbnail';
	} else {
		$type = ''; //important
	}

	/**
	 * Filters the staff single post media type.
	 *
	 * @param string $type
	 */
	$type = (string) apply_filters( 'wpex_staff_single_media_type', $type );

	return $type;
}

/**
 * Returns thumbnail HTML for the staff posts.
 *
 * @since 2.0.0
 */
function wpex_get_staff_post_thumbnail( $args = '' ) {
	$defaults = array(
		'size'          => 'staff_post',
		'class'         => 'staff-single-media-img wpex-align-middle',
		'schema_markup' => true,
	);

	$args = wp_parse_args( $args, $defaults );

	$args = (array) apply_filters( 'wpex_get_staff_post_thumbnail_args', $args ); // deprecated 5.5.3

	/**
	 * Filters the staff single post thumbnail args
	 *
	 * @param array $args
	 */
	$args = (array) apply_filters( 'wpex_staff_post_thumbnail_args', $args );

	return wpex_get_post_thumbnail( $args );
}

/**
 * Staff single media class.
 */
function wpex_staff_single_media_class() {
	$class = array(
		'single-media',
		'wpex-mb-20',
	);

	/**
	 * Filters the staff single media element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_single_media_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Staff single header class.
 */
function wpex_staff_single_header_class() {
	$class = array(
		'single-header',
		'wpex-mb-20',
	);

	/**
	 * Filters the staff single header element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_single_header_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Staff single title class.
 */
function wpex_has_staff_single_title_position() {
	$check = false;

	if ( get_theme_mod( 'staff_single_header_position', true ) && wpex_has_staff_member_position() ) {
		$check = true;
	}

	/**
	 * Filters whether the single staff title displays the staff position or not.
	 *
	 * @param array $class
	 */
	$check = (bool) apply_filters( 'wpex_has_staff_single_title_position', $check );

	return $check;
}

/**
 * Staff single title class.
 */
function wpex_staff_single_title_class() {
	$class = array(
		'single-post-title',
		'entry-title',
		'wpex-text-3xl',
	);

	if ( wpex_has_staff_single_title_position() ) {
		$class[] = 'wpex-m-0';
	}

	/**
	 * Filters the staff single title element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_single_title_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Staff single position class.
 */
function wpex_staff_single_position_class() {
	$class = array(
		'single-staff-position',
		'wpex-text-md',
		'wpex-text-gray-500',
	);

	/**
	 * Filters the staff single position element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_single_position_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Staff single content class.
 */
function wpex_staff_single_content_class() {
	$class = array(
		'single-content',
		'wpex-mt-20',
		'entry',
	);

	if ( ! wpex_has_post_wpbakery_content( get_the_ID() ) ) {
		$class[] = 'wpex-mb-40';
	}

	$class[] = 'wpex-clr';

	/**
	 * Filters the staff single content element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_single_content_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Staff single comments class.
 */
function wpex_staff_single_comments_class() {
	$class = array(
		'single-comments',
		'wpex-mb-40',
		'wpex-clr',
	);

	/**
	 * Filters the staff single comments element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_single_comments_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/*-------------------------------------------------------------------------------*/
/* [ Related ]
/*-------------------------------------------------------------------------------*/

/**
 * Return staff single related query.
 */
function wpex_staff_single_related_query() {
	$post_id = get_the_ID();

	// Return if disabled via post meta.
	if ( wpex_validate_boolean( get_post_meta( $post_id, 'wpex_disable_related_items', true ) ) ) {
		return false;
	}

	// Posts count.
	$posts_count = (int) get_theme_mod( 'staff_related_count', 3 );

	// Return if count is empty or 0.
	if ( ! $posts_count ) {
		return false;
	}

	// Related query arguments.
	$args = array(
		'post_type'      => 'staff',
		'posts_per_page' => $posts_count,
		'order'          => get_theme_mod( 'staff_related_order', 'desc' ),
		'orderby'        => get_theme_mod( 'staff_related_orderby', 'date' ),
		'post__not_in'   => array( $post_id ),
		'no_found_rows'  => true,
	);

	// Related by taxonomy.
	if ( apply_filters( 'wpex_related_in_same_cat', true ) ) {

		// Add categories to query.
		$related_taxonomy = get_theme_mod( 'staff_related_taxonomy', 'staff_category' );

		// Generate related by taxonomy args.
		if ( 'null' !== $related_taxonomy && taxonomy_exists( $related_taxonomy ) ) {

			$terms = '';

			$primary_term = wpex_get_post_primary_term( $post_id, $related_taxonomy );

			if ( $primary_term ) {

				$terms = array( $primary_term->term_id );

			} else {

				$get_terms = get_the_terms( $post_id, $related_taxonomy );

				if ( $get_terms && ! is_wp_error( $get_terms ) ) {
					$terms = wp_list_pluck( $get_terms, 'term_id' );
				}

			}

			if ( $terms ) {

				$args['tax_query'] = array(
					'relation' => 'AND',
					array(
						'taxonomy' => $related_taxonomy,
						'field'    => 'term_id',
						'terms'    => $terms,
					)
				);

			}

		}

	}

	/**
	 * Filters the staff related query args.
	 *
	 * @param array $class
	 */
	$args = (array) apply_filters( 'wpex_related_staff_args', $args );

	if ( $args ) {
		return new wp_query( $args );
	}
}

/**
 * Display staff single related heading.
 */
function wpex_staff_single_related_heading() {
	$heading = wpex_get_translated_theme_mod( 'staff_related_title' );

	if ( ! $heading ) {
		$heading = esc_html__( 'Related Staff', 'total' );
	}

	wpex_heading( array(
		'tag'           => get_theme_mod( 'related_heading_tag' ) ?: 'h3',
		'content'		=> $heading,
		'classes'		=> array( 'related-staff-posts-heading' ),
		'apply_filters'	=> 'staff_related',
	) );
}

/**
 * Staff single related class.
 */
function wpex_staff_single_related_class() {
	$class = array(
		'related-staff-posts',
		'wpex-mb-20',
	);

	if ( 'full-screen' === wpex_content_area_layout() ) {
		$class[] = 'container';
	}

	$class[] = 'wpex-clr';

	/**
	 * Filters the staff single related element class.
	 *
	 * @param array $class
	 */
	$class = (array) apply_filters( 'wpex_staff_single_related_class', $class );

	if ( $class ) {
		echo 'class="' . esc_attr( implode( ' ', array_unique( $class ) ) ) . '"';
	}
}

/**
 * Staff single related row class.
 * @todo deprecate wpex_related_staff_row_classes filter
 */
function wpex_staff_single_related_row_class() {
	$classes = array(
		'wpex-row',
		'wpex-clr'
	);

	if ( $gap = get_theme_mod( 'staff_related_gap' ) ) {
		$classes[] = wpex_gap_class( $gap );
	}

	/**
	 * Filters the staff single related row element class.
	 *
	 * @param array $class
	 */
	$classes = (array) apply_filters( 'wpex_staff_single_related_row_class', $classes );

	if ( $classes ) {
		echo 'class="' . esc_attr( apply_filters( 'wpex_related_staff_row_classes', implode( ' ', $classes ) ) ) . '"';
	}
}



/*-------------------------------------------------------------------------------*/
/* [ Social ]
/*-------------------------------------------------------------------------------*/

/**
 * Outputs the staff social options.
 */
if ( ! function_exists( 'wpex_get_staff_social' ) ) {
	function wpex_get_staff_social( $atts = NULL ) {
		extract( shortcode_atts( array(
			'before'             => '',
			'after'              => '',
			'post_id'            => '',
			'font_size'          => '',
			'css_animation'      => '',
			'css'                => '',
			'inline_style'       => '',
			'icon_margin'        => '',
			'spacing'            => '5',
			'animation_delay'    => '',
			'animation_duration' => '',
			'show_icons'         => wp_validate_boolean( get_theme_mod( 'staff_social_show_icons', true ) ),
			'labels_only'        => false,
			'format'             => '', // added in 5.4.3 @todo make "flex" format the default!
			'margin_top'         => 10, // this is fallback from old styles.
			'separator'          => '',
			'style'              => get_theme_mod( 'staff_social_default_style', 'minimal-round' ),
			'link_target'        => get_theme_mod( 'staff_social_link_target', 'blank' ),
		), $atts, 'staff_social' ) );

		// Define output.
		$output = '';

		// New flex format added in 5.4.3.
		if ( $format && 'flex' === $format ) {
			$has_flex_format = true;
		} else {
			$has_flex_format = false;
		}

		// Only show labels (added in 5.4.3)
		if ( $labels_only ) {
			$show_icons = false;
		}

		// Get social profiles array.
		$profiles = wpex_staff_social_array();

		// Check for dynamic ID.
		if ( empty( $post_id ) ) {
			$query_var = get_query_var( 'wpex_current_post_id' );
			if ( $query_var ) {
				$post_id = $query_var;
			}
		}

		// Make sure post_id is defined.
		if ( ! $post_id ) {
			$post_id = get_the_ID();
		}

		// Convert icon margin to style.
		if ( $icon_margin && function_exists( 'vcex_inline_style' ) ) {
			$icon_margin = vcex_inline_style( array(
				'margin' => $icon_margin,
			), false );
		}

		// Parse style to return correct classname.
		if ( $show_icons ) {
			$style_class = wpex_get_social_button_class( $style );
		} else {
			$style_class = '';
		}

		// Wrap classes.
		$wrap_classes = array(
			'staff-social',
			'wpex-social-btns',
		);

		if ( $margin_top && '0px' !== $margin_top && '0' !== $margin_top ) {
			$wrap_classes[] = 'wpex-mt-' . absint( $margin_top );
		}

		if ( $has_flex_format ) {
			$wrap_classes[] = 'wpex-flex wpex-flex-wrap';
			if ( $spacing ) {
				$wrap_classes[] = 'wpex-gap-' . sanitize_html_class( $spacing );
			}
		} else {
			$wrap_classes[] = 'wpex-last-mr-0';
		}

		if ( $css ) {
			$wrap_classes[] = vc_shortcode_custom_css_class( $css );
		}
		if ( $css_animation && 'none' != $css_animation && function_exists( 'vcex_get_css_animation' ) ) {
			$wrap_classes[] = vcex_get_css_animation( $css_animation );
		}

		// Font size.
		if ( $font_size ) {
			if ( $font_size && array_key_exists( $font_size, wpex_utl_font_sizes() ) ) {
				$font_size = 'var(--wpex-text-' . esc_attr( $font_size ) . ')';
			} else {
				$font_size = wpex_sanitize_data( $font_size, 'font_size' );
			}
			if ( $font_size ) {
				$inline_style .= ' font-size:' . esc_attr( $font_size ) . ';';
			}
		}

		// Animation delay.
		if ( $animation_delay ) {
			$inline_style .= ' animation-delay:' . floatval( $animation_delay ) . 's;';
		}

		// Animation duration.
		if ( $animation_duration ) {
			$inline_style .= ' animation-duration:' . floatval( $animation_duration ) . 's;';
		}

		// Wrap attributes.
		$wrap_attrs = array(
			'class' => array_map( 'esc_attr', $wrap_classes ),
			'style' => trim( $inline_style ),
		);

		// Before output.
		if ( $before ) {
			$output .= $before;
		}

		// Start output.
		$output .= '<div ' . wpex_parse_attrs( $wrap_attrs ) . '>';

			// Loop through social options.
			$count = 0;
			foreach ( $profiles as $profile ) :

				// Get meta.
				$meta = $profile['meta'];

				// Get link.
				$meta_value = get_post_meta( $post_id, $meta, true );

				// Continue if no link is set.
				if ( ! $meta_value ) {
					continue;
				}

				// Add to counter.
				$count ++;

				// Add separator.
				if ( $separator && $count > 1 ) {
					$output .= '<span class="staff-social__separator';
						if ( ! $has_flex_format ) {
							$output .= ' wpex-mr-' . sanitize_html_class( $spacing );
						}
					$output .= '">' . wp_strip_all_tags( trim( $separator ) ) . '</span>';
				}

				// Add "mailto" for emails.
				if ( 'wpex_staff_email' === $meta && is_email( $meta_value ) ) {
					$url = 'mailto:' . $meta_value;
				}

				// Add "callto" to skype.
				elseif ( 'wpex_staff_skype' === $meta ) {
					if ( strpos( $meta_value, 'skype' ) === false ) {
						$url = str_replace( 'callto:', '', $meta_value );
						$url = 'callto:' . $url;
					}
				}

				// Add "tel" for phones.
				elseif ( 'wpex_staff_phone_number' === $meta ) {
					if ( strpos( $meta_value, 'callto' ) === false ) {
						$url = str_replace( 'tel:', '', $meta_value );
						$url = 'tel:' . $url;
					}
				} else {
					$url = esc_url( $meta_value );
				}

				// Link classes.
				$link_classes = array(
					'wpex-' . sanitize_html_class( str_replace( '_', '-', $profile['key'] ) ),
				);

				if ( $style_class ) {
					$link_classes[] = $style_class;
				}

				if ( $has_flex_format ) {
					$link_classes[] = '';
				} else {
					$link_classes[] = 'wpex-mr-' . sanitize_html_class( $spacing );
					$link_classes[] = 'wpex-mt-' . sanitize_html_class( $spacing );
				}

				if ( ! $show_icons && ! $has_flex_format ) {
					$link_classes[] = 'wpex-inline-block';
				}

				// Link attributes.
				$link_attrs  = array(
					'href'   => $url,
					'class'  => $link_classes,
					'target' => ( 'wpex_staff_email' === $meta ) ? '' : $link_target,
					'style'  => $icon_margin,
				);

				$output .= '<a '. wpex_parse_attrs( $link_attrs ) .'>';

				$icon_class = $profile['icon_class'];

					if ( $show_icons ) {

						if ( str_starts_with( $icon_class, 'ticon' ) ) {
							$output .= wpex_get_theme_icon_html( $icon_class );
						} else {
							$output .= '<span class="' . esc_attr( $icon_class ) . '" aria-hidden="true"></span>';
						}

						$output .= '<span class="screen-reader-text">' . esc_html( $profile['label'] ) . '</span>';

					} else {

						$label = $profile['label'];

						if ( ! $labels_only && in_array( $profile['key'], array(
								'telephone',
								'fax',
								'phone_number',
								'phone',
								'email',
								'website'
							) )
						) {

							$label = $meta_value;

						} else {

							if ( $labels_only ) {
								switch ( $profile['key'] ) {
									case 'telephone':
									case 'phone':
									case 'phone_number':
										$label = esc_html__( 'Phone', 'total' );
										break;
								}
							}

							/**
							 * Filters the staff social label.
							 *
							 * @param string $label
							 * @param string $key
							 */
							$label = apply_filters( 'wpex_staff_social_item_label', $label, $profile['key'] );

						}

						$output .= esc_html( $label );

					}


				$output .= '</a>';

			endforeach; // End profiles loop.

		// End output.
		$output .= '</div>';

		// After output.
		if ( $after ) {
			$output .= $after;
		}

		return $output;
	}
}

/*-------------------------------------------------------------------------------*/
/* [ Cards ]
/*-------------------------------------------------------------------------------*/

/**
 * Staff Card Entry.
 */
function wpex_staff_entry_card() {
	$instance = wpex_get_loop_instance();

	if ( 'related' === $instance ) {
		$card_style = get_theme_mod( 'staff_related_entry_card_style' );
	} else {
		$term_meta_check = wpex_get_term_meta( '', 'wpex_entry_card_style', true );
		if ( ! empty( $term_meta_check ) ) {
			$card_style = $term_meta_check;
		} else {
			$card_style = get_theme_mod( 'staff_entry_card_style' );
		}
	}

	$card_style = apply_filters( 'wpex_staff_entry_card_style', $card_style );

	if ( ! $card_style ) {
		return false;
	}

	$args = array(
		'style'          => $card_style,
		'post_id'        => get_the_ID(),
		'thumbnail_size' => ( 'related' === $instance ) ? 'staff_related' : 'staff_entry',
		'excerpt_length' => wpex_staff_entry_excerpt_length(),
	);

	if ( $overlay_style = wpex_overlay_style() ) {
		$args['thumbnail_overlay_style'] = $overlay_style;
	}

	if ( $hover_style = get_theme_mod( 'staff_entry_image_hover_animation', null ) ) {
		$args['thumbnail_hover'] = $hover_style;
	}

	/**
	 * Filters the staff entry card args.
	 *
	 * @param array $args
	 */
	$args = (array) apply_filters( 'wpex_staff_entry_card_args', $args );

	wpex_card( $args );

	return true;
}

/*-------------------------------------------------------------------------------*/
/* [ Deprecated ]
/*-------------------------------------------------------------------------------*/

/**
 * Staff Overlay.
 *
 * @todo deprecate
 */
if ( ! function_exists( 'wpex_get_staff_overlay' ) ) {
	function wpex_get_staff_overlay( $post_id = NULL ) {
		if ( ! $post_id ) {
			$post_id = get_the_ID();
		}

		$position = wpex_get_staff_member_position( $post_id );

		if ( ! $position ) {
			return;
		}

		?>
		<div class="staff-entry-position"><span><?php echo esc_html( $position ); ?></span></div>
		<?php
	}
}