<?php
namespace TotalTheme;

use TotalTheme\Integration\Yoast_SEO\Helpers as Yoast_SEO_Helpers;

defined( 'ABSPATH' ) || exit;

/**
 * Error 404 Class.
 *
 * @package Total WordPress theme
 * @subpackage Framework
 * @version 5.6.1
 */
class Error_404 {

	/**
	 * Class instance.
	 */
	private static $instance;

	/**
	 * Class instance.
	 */
	protected static $template_id;

	/**
	 * Create or retrieve the instance of Custom_Actions.
	 */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new self();
		}
		return static::$instance;
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		if ( ! get_theme_mod( 'custom_404_enable', true ) ) {
			return;
		}
		if ( wpex_is_request( 'admin' ) ) {
			$this->admin_init();
		}
		if ( wpex_is_request( 'frontend' ) ) {
			add_action( 'init', [ $this, 'init' ] );
		}
	}

	/**
	 * Admin hooks.
	 */
	public function admin_init() {
		add_action( 'admin_menu', [ $this, 'add_submenu_page' ] );
		add_action( 'admin_init', [ $this, 'register_page_options' ] );
	}

	/**
	 * Front-end hooks.
	 */
	public function init() {
		if ( get_theme_mod( 'error_page_redirect', false ) ) {
			add_filter( 'template_redirect', [ $this, 'redirect_home' ] );
		}
		if ( $this->get_template_id() ) {
			add_filter( 'wpex_current_post_id', [ $this, 'post_id' ] );
			add_filter( 'wp_robots', [ $this, 'add_noindex_tag' ] );
		}
		if ( did_action( 'wpseo_loaded' ) && $template_id = $this->get_template_id() ) {
			new Yoast_SEO_Helpers\Exclude_From_Sitemap( $template_id );
		}
	}

	/**
	 * Add sub menu page for the custom CSS input.
	 */
	public function add_submenu_page() {
		add_submenu_page(
			WPEX_THEME_PANEL_SLUG,
			esc_html__( 'Custom 404', 'total' ),
			esc_html__( 'Custom 404', 'total' ),
			'edit_theme_options',
			WPEX_THEME_PANEL_SLUG . '-404',
			[ $this, 'create_admin_page' ]
		);
	}

	/**
	 * Function that will register admin page options.
	 */
	public function register_page_options() {

		// Register settings.
		register_setting(
			'wpex_error_page',
			'error_page',
			array(
				'sanitize_callback' => [ $this, 'save_options' ],
				'default' => null,
			)
		);

		// Add main section to our options page.
		add_settings_section(
			'wpex_error_page_main',
			false,
			[ $this, 'section_main_callback' ],
			'wpex-custom-error-page-admin'
		);

		// Redirect field.
		add_settings_field(
			'redirect',
			esc_html__( 'Redirect 404\'s', 'total' ),
			[ $this, 'redirect_field_callback' ],
			'wpex-custom-error-page-admin',
			'wpex_error_page_main'
		);

		// Custom Page ID.
		add_settings_field(
			'error_page_id',
			esc_html__( 'Custom 404 Page', 'total' ),
			[ $this, 'content_id_field_callback' ],
			'wpex-custom-error-page-admin',
			'wpex_error_page_main'
		);

		// Title field.
		add_settings_field(
			'error_page_title',
			esc_html__( '404 Page Title', 'total' ),
			[ $this, 'title_field_callback' ],
			'wpex-custom-error-page-admin',
			'wpex_error_page_main'
		);

		// Content field.
		add_settings_field(
			'error_page_text',
			esc_html__( '404 Page Content', 'total' ),
			[ $this, 'content_field_callback' ],
			'wpex-custom-error-page-admin',
			'wpex_error_page_main'
		);

	}

	/**
	 * Save options.
	 */
	public function save_options( $options ) {

		// Set theme mods
		if ( isset( $options['redirect'] ) ) {
			set_theme_mod( 'error_page_redirect', 1 );
		} else {
			remove_theme_mod( 'error_page_redirect' );
		}

		if ( ! empty( $options['title'] ) ) {
			set_theme_mod( 'error_page_title', $options['title'] );
		} else {
			remove_theme_mod( 'error_page_title' );
		}

		if ( ! empty( $options['text'] ) ) {
			set_theme_mod( 'error_page_text', $options['text'] );
		} else {
			remove_theme_mod( 'error_page_text' );
		}

		if ( ! empty( $options['content_id'] ) ) {
			set_theme_mod( 'error_page_content_id', $options['content_id'] );
		} else {
			remove_theme_mod( 'error_page_content_id' );
		}

		// Don't actually save as an option since we are using mods.
		return;
	}

	/**
	 * Main Settings section callback.
	 */
	public function section_main_callback( $options ) {
		// Leave blank
	}

	/**
	 * Fields callback functions.
	 */

	// Redirect field
	public function redirect_field_callback() {
		$val = get_theme_mod( 'error_page_redirect' );
		echo '<input type="checkbox" name="error_page[redirect]" id="error-page-redirect" value="'. esc_attr( $val ) .'" '. checked( $val, true, false ) .'> ';
		echo '<span class="description">'. esc_html__( 'Automatically 301 redirect all 404 errors to your homepage.', 'total' ) .'</span>';
	}

	// Custom Error Page ID.
	public function content_id_field_callback() {
		$page_id = get_theme_mod( 'error_page_content_id' );

		wp_dropdown_pages( [
			'echo'             => true,
			'selected'         => $page_id,
			'name'             => 'error_page[content_id]',
			'id'               => 'error-page-content-id',
			'class'            => 'wpex-chosen',
			'show_option_none' => esc_html__( 'None', 'total' ),
		] ); ?>

		<br>

		<p class="description"><?php esc_html_e( 'Select a custom page to replace your default 404 page content.', 'total' ) ?></p>

		<?php if ( $page_id ) { ?>

			<br>

			<div id="wpex-footer-builder-edit-links">

				<a href="<?php echo admin_url( 'post.php?post='. $page_id .'&action=edit' ); ?>" class="button" target="_blank" rel="noopener noreferrer">
					<?php esc_html_e( 'Backend Edit', 'total' ); ?>
				</a>

				<?php if ( WPEX_VC_ACTIVE ) { ?>

					<a href="<?php echo admin_url( 'post.php?vc_action=vc_inline&post_id='. $page_id .'&post_type=page' ); ?>" class="button" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Frontend Edit', 'total' ); ?></a>

				<?php } ?>

				<a href="<?php echo esc_url( get_permalink( $page_id ) ); ?>" class="button" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Preview', 'total' ); ?></a>

			</div>

		<?php } ?>

	<?php }

	// Title field.
	public function title_field_callback() { ?>
		<input type="text" name="error_page[title]" id="error-page-title" value="<?php echo get_theme_mod( 'error_page_title' ); ?>">
		<p class="description"><?php esc_html_e( 'Enter a custom title for your 404 page.', 'total' ) ?></p>
	<?php }

	// Content field.
	public function content_field_callback() {
		wp_editor( get_theme_mod( 'error_page_text' ), 'error_page_text', array(
			'textarea_name' => 'error_page[text]'
		) );
	}

	/**
	 * Settings page output.
	 */
	public function create_admin_page() {
		wp_enqueue_style( 'wpex-chosen' );
		wp_enqueue_script( 'wpex-chosen' );

		wp_enqueue_script(
			'wpex-custom-404',
			wpex_asset_url( 'js/dynamic/admin/wpex-custom-404.min.js' ),
			array( 'jquery' ),
			WPEX_THEME_VERSION
		);

		?>

		<div class="wrap">
			<h1><?php esc_html_e( 'Custom 404', 'total' ); ?></h1>
			<form method="post" action="options.php">
				<?php settings_fields( 'wpex_error_page' ); ?>
				<?php do_settings_sections( 'wpex-custom-error-page-admin' ); ?>
				<?php submit_button(); ?>
			</form>
		</div>
	<?php }

	/**
	 * Hooks into "wp_robots" to add the noindex tag to the custom 404 page.
	 */
	public function add_noindex_tag( $robots ) {
		$template_id = $this->get_template_id();
		if ( $template_id
			&& is_page( $template_id )
			&& $template_id === get_queried_object_id() // just an extra check to be extra safe!
		) {
			$robots['noindex'] = true;
		}
		return $robots;
	}

	/**
	 * Redirect all pages to the under cronstruction page if user is not logged in.
	 */
	public function redirect_home() {
		if ( is_404() ) {
			wp_redirect( esc_url( home_url( '/' ) ), 302 );
			exit();
		}
	}

	/**
	 * Custom VC CSS for 404 custom page design.
	 */
	public function post_id( $post_id ) {
		if ( is_404() ) {
			$error_page = $this->get_template_id();
			if ( $error_page ) {
				$post_id = $error_page;
			}
		}
		return $post_id;
	}

	/**
	 * Returns inline CSS for the 404 page.
	 */
	public function get_template_id() {
		if ( ! is_null( self::$template_id ) ) {
			return self::$template_id;
		}
		$template_id = absint( get_theme_mod( 'error_page_content_id' ) );
		if ( ! $template_id ) {
			self::$template_id = 0;
			return self::$template_id;
		}
		$template_id   = wpex_parse_obj_id( $template_id, 'page' );
		$template_post = get_post( $template_id );
		if ( is_a( $template_post, 'WP_Post' ) && 'publish' === get_post_status( $template_post ) ) {
			self::$template_id = absint( $template_id );
			return $template_id;
		}
	}

	/**
	 * Returns inline CSS for the 404 page.
	 */
	protected function get_template_css( $template_id ) {
		$css = '';
		$post_css = get_post_meta( $template_id, '_wpb_post_custom_css', true );
		$post_css = apply_filters( 'vc_post_custom_css', $post_css, $template_id );
		if ( $post_css ) {
			$css .= $post_css;
		}
		$shortcode_css = get_post_meta( $template_id, '_wpb_shortcodes_custom_css', true );
		$shortcode_css = apply_filters( 'vc_shortcodes_custom_css', $shortcode_css, $template_id );
		if ( $shortcode_css ) {
			$css .= $shortcode_css;
		}
		if ( $css ) {
			return '<style>' . wp_strip_all_tags( $css ) . '</style>';
		}
	}

	/**
	 * Gets the content for the custom 404 page.
	 *
	 * @return string
	 */
	protected function get_the_content() {
		$template_id = $this->get_template_id();

		// Display template content.
		if ( $template_id && $current_id = wpex_get_current_post_id() ) {
			$post         = get_post( $current_id ); // get post
			$content      = wpex_parse_vc_content( $post->post_content ); // remove weird p tags and extra code
			$content      = wp_kses_post( $content ); // security
			$content      = do_shortcode( $content ); // parse shortcodes and echo
			$template_css = $this->get_template_css( $current_id );
			if ( $template_css ) {
				$content = $template_css . $content;
			}
			return $content;
		}

		// Custom Text.
		$error_text = trim( wpex_get_translated_theme_mod( 'error_page_text' ) );

		if ( $error_text ) {
			return '<div class="custom-error404-content wpex-clr">' . wpex_the_content( $error_text, 'error404' ) .'</div>';
		}

		// Default text.
		return '<div class="error404-content wpex-text-center wpex-py-30 wpex-clr"><h1 class="error404-content-heading wpex-m-0 wpex-mb-10 wpex-text-6xl">' . esc_html__( 'Sorry, this page could not be found.', 'total' ) . '</h1><div class="error404-content-text wpex-text-md wpex-last-mb-0">' . esc_html__( 'The page you are looking for doesn\'t exist, no longer exists or has been moved.', 'total' ) . '</div></div>';
	}

	/**
	 * Renders the custom 404 page.
	 *
	 * @return string
	 */
	public function render() {
		// @codingStandardsIgnoreLine
		echo $this->get_the_content();
	}

	/**
	 * Returns custom 404 page content.
	 *
	 * @deprecated 5.6.1
	 */
	public static function get_content() {
		ob_start();
			self::instance()->render();
		return ob_get_clean();
	}

}