<?php
namespace TotalTheme;

use TotalTheme\Header\Core as Header;
use TotalTheme\Header\Logo as Header_Logo;
use TotalTheme\Header\Overlay as Overlay_Header;
use TotalTheme\Header\Sticky as Sticky_Header;
use TotalTheme\Mobile\Menu as Mobile_Menu;

defined( 'ABSPATH' ) || exit;

/**
 * Advanced frontend styles based on user settings.
 *
 * @package TotalTheme
 * @version 5.6.1
 */
final class Advanced_Styles {

	/**
	 * CSS var.
	 */
	private $css;

	/**
	 * Class instance.
	 */
	public static $instance;

	/**
	 * Create or retrieve the class instance.
	 */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new self();
			static::$instance->init_hooks();
		}
		return static::$instance;
	}

	/**
	 * Constructor.
	 */
	public function init_hooks() {
		add_filter( 'wpex_head_css', [ $this, 'update_head_css' ], 999 );
	}

	/**
	 * Hooks into wpex_head_css to add custom css to the <head> tag.
	 */
	public function update_head_css( $head_css ) {
		$this->generate_css();

		if ( ! empty( $this->css ) ) {
			$head_css .= '/*ADVANCED STYLING CSS*/' . $this->css;
		}

		return $head_css;
	}

	/**
	 * Generates css.
	 */
	private function generate_css() {
		$methods = [
			'header_background',
			'overlay_header',
			'logo_mobile_side_margin',
			'logo_height',
			'page_header_title',
			'footer_background_image',
			'footer_callout_background_image',
			'blockquote_border',
			'bold_font_weight',
			'mobile_menu',
		];

		if ( WPEX_VC_ACTIVE ) {
			$methods[] = 'vc_inline_shrink_sticky_header_height';
			if ( get_option( 'wpb_js_use_custom' ) ) {
				$methods[] = 'vc_column_gutter';
			}
		}

		if ( WPEX_WOOCOMMERCE_ACTIVE ) {
			$methods[] = 'woo_full_width_add_to_cart';
			$methods[] = 'woo_customizer_hide_elements';
			$methods[] = 'woo_hide_quantity';
		}

		foreach ( $methods as $method ) {
			$method_css = $this->$method();
			if ( $method_css ) {
				$this->css .= $method_css;
			}
		}

	}

	/**
	 * Header background.
	 */
	private function header_background() {
		$header_bg = Header::get_background_image_url();
		if ( $header_bg ) {
			return '#site-header{background-image:url(' . esc_url( $header_bg ) . ');}';
		}
	}

	/**
	 * Overlay header.
	 */
	private function overlay_header() {
		$css        = '';
		$post_id    = wpex_get_current_post_id();
		$is_enabled = Overlay_Header::is_enabled();

		if ( ! $is_enabled ) {
			return;
		}

		$breakpoint = Overlay_Header::get_breakpoint();

		if ( $post_id && wpex_has_post_meta( 'wpex_overlay_header' )) {

			// Custom overlay header font size.
			$overlay_header_font_size = get_post_meta( $post_id, 'wpex_overlay_header_font_size', true );

			if ( $overlay_header_font_size ) {
				$css .= '#site-navigation, #site-navigation .main-navigation-ul a{font-size:' . intval( $overlay_header_font_size ) . 'px;}';
			}

			// Apply overlay header background color.
			// Note we use background and not background-color.
			$overlay_header_bg = get_post_meta( $post_id, 'wpex_overlay_header_background', true );

			if ( $overlay_header_bg ) {
				$css .= '#site-header.overlay-header.dyn-styles{background:' . esc_attr( $overlay_header_bg ) . '; }';
			}

		}

		if ( $breakpoint && is_int( $breakpoint ) ) {
			$css = '@media only screen and (min-width:' . ( $breakpoint + 1 )  . 'px){' . $css . '}';

			// Hide overlay logo.
			$css .= '@media only screen and (max-width:' . ( $breakpoint )  . 'px){.overlay-header--responsive .logo-img--overlay{display: none;}}';
		}

		return $css;
	}

	/**
	 * Logo mobile side margin.
	 */
	private function logo_mobile_side_margin() {
		$css          = '';
		$header_style = Header::Style();

		if ( 'seven' === $header_style
			|| 'eight' === $header_style
			|| 'nine' === $header_style
			|| 'ten' === $header_style
		) {
			return;
		}

		$margin_escaped = absint( get_theme_mod( 'logo_mobile_side_offset' ) );

		if ( ! empty( $margin_escaped ) ) {

			$mm_breakpoint = Mobile_Menu::breakpoint();

			if ( $mm_breakpoint < 9999 ) {
				$css .= '@media only screen and (max-width:' . $mm_breakpoint . 'px) {';
			}

			$css .= 'body.has-mobile-menu #site-logo {';
				if ( is_rtl() ) {
					$css .= 'margin-left:' . $margin_escaped . 'px;';
				} else {
					$css .= 'margin-right:' . $margin_escaped . 'px;';
				}
			$css .= '}';

			if ( $mm_breakpoint < 9999 ) {
				$css .= '}';
			}

		}

		return $css;
	}

	/**
	 * Custom logo height.
	 */
	private function logo_height() {
		$logo_is_svg = Header_Logo::is_image_svg();
		if ( $logo_is_svg || get_theme_mod( 'apply_logo_height', false ) ) {
			$height = absint( get_theme_mod( 'logo_height' ) );
			$height_prop = 'max-height';
			if ( Header::has_fixed_height() || $logo_is_svg ) {
				$height_prop = 'height';
			}
			if ( $height ) {
				return '#site-logo .logo-img{' . $height_prop . ':' . esc_attr( $height ) . 'px;width:auto;}'; // auto width needed for responsiveness.
			}
		}
	}

	/**
	 * Shrink header height.
	 *
	 * This is used to provide consistency with the shrink header logo when using the front-end builder
	 * since the sticky header is disabled when using the builder.
	 *
	 * @todo perhaps we can target a different classname in the default CSS that gets added on the front-end only?
	 */
	private function vc_inline_shrink_sticky_header_height() {
		if ( ! wpex_vc_is_inline() || Header::has_flex_container() ) { // important not needed for flex header!!
			return;
		}

		$shrink_header_style = Sticky_Header::style();

		if ( 'shrink' === $shrink_header_style || 'shrink_animated' === $shrink_header_style ) {
			return '#site-logo .logo-img{max-height:var(--wpex-site-header-shrink-start-height, 60px)!important;width:auto;}';
		}
	}

	/**
	 * Page header title.
	 */
	private function page_header_title() {
		if ( ! wpex_has_post_meta( 'wpex_post_title_style' ) ) {
			$page_header_bg = wpex_page_header_background_image(); // already passed through wpex_get_image_url
			if ( $page_header_bg ) {
				return '.page-header.has-bg-image{background-image:url(' . esc_url( $page_header_bg ) . ');}';
			}
		}
	}

	/**
	 * Footer background.
	 */
	private function footer_background_image() {
		$background = wpex_get_image_url( get_theme_mod( 'footer_bg_img' ) );
		if ( $background ) {
			return '#footer{background-image:url(' . esc_url( $background ) . ');}';
		}
	}

	/**
	 * Footer callout background.
	 */
	public function footer_callout_background_image() {
		$background = wpex_get_image_url( get_theme_mod( 'footer_callout_bg_img' ) );
		if ( $background ) {
			return '#footer-callout-wrap{background-image:url(' . esc_url( $background ) . ');}';
		}
	}

	/**
	 * Define Visual Composer gutter.
	 */
	private function vc_column_gutter() {
		$custom_gutter = get_option( 'wpb_js_gutter' );
		if ( $custom_gutter ) {
			return ':root{--wpex-vc-gutter:' . absint( $custom_gutter ) . 'px}';
		}
	}

	/**
	 * Adds border to the blockquote element.
	 */
	private function blockquote_border() {
		$border_width = get_theme_mod( 'blockquote_border_width' );
		if ( $border_width ) {
			return 'blockquote{background:none;border-width: 0 0 0 ' . absint( $border_width ) . 'px;border-color:var(--wpex-accent);border-style:solid;padding-left:25px;}';
		}
	}

	/**
	 * Alters the --wpex-bold CSS var.
	 */
	private function bold_font_weight() {
		$value = get_theme_mod( 'bold_font_weight' );
		if ( $value && is_numeric( $value ) ) {
			return ':root{--wpex-bold:' . wp_strip_all_tags( $value ) . ';}';
		}
	}

	/**
	 * Full Width add to cart button.
	 */
	private function woo_full_width_add_to_cart() {
		if ( get_theme_mod( 'woo_product_add_to_cart_full_width' ) ) {
			return '.woocommerce .product .summary .single_add_to_cart_button, .woocommerce .product .summary .added_to_cart,.product .add_to_cart_button,.product .added_to_cart,.product-actions .button{width:100%;}.woocommerce .summary .quantity{margin-bottom:20px;}';
		}
	}

	/**
	 * Hide WooCommerce Elements while in the Customizer.
	 */
	private function woo_customizer_hide_elements() {
		if ( ! is_customize_preview() ) {
			return;
		}
		$css = '';
		if ( function_exists( 'is_product' ) && is_product() ) {
			$product_title = get_theme_mod( 'woo_shop_single_title' );
			if ( $product_title && false !== strpos( $product_title, '{{title}}' ) ) {
				$css .= '.woocommerce .summary .single-post-title{display:none !important;}';
			}
		}
		if ( ! get_theme_mod( 'woo_shop_sort', true ) ) {
			$css .= '.woocommerce .woocommerce-ordering{display:none !important;}';
		}
		if ( ! get_theme_mod( 'woo_shop_result_count', true ) ) {
			$css .= '.woocommerce .woocommerce-result-count{display:none !important;}';
		}
		if ( ! get_theme_mod( 'woo_product_meta', true ) ) {
			$css .= '.woocommerce .product_meta{display:none !important;}';
		}
		return $css;
	}

	/**
	 * Hide WooCommerce Quantity.
	 */
	private function woo_hide_quantity() {
		if ( 'disabled' === get_theme_mod( 'woo_quantity_buttons_style' ) ) {
			return '.summary .quantity, .vcex-wc-template-part--single-product__add-to-cart .quantity{display:none;}';
		}
	}

	/**
	 * Mobile menu tweaks.
	 */
	private function mobile_menu() {
		$css = '';
		if ( 'toggle' === Mobile_Menu::style() ) {
			$border_bottom = get_theme_mod( 'mobile_menu_toggle_has_border_bottom' );
			if ( wp_validate_boolean( $border_bottom ) ) {
				$css .= '.mobile-toggle-nav-ul { border-bottom: 1px solid var(--wpex-border-main); }';
				$css .= '.mobile-toggle-nav-search { margin-top: 20px; }';
			}
			$border_top = get_theme_mod( 'mobile_menu_toggle_has_border_top', true );
			if ( false === wp_validate_boolean( $border_top ) ) {
				$css .= '.mobile-toggle-nav-ul > li:first-child > a { border-top: 0; }';
			}
		}
		return $css;
	}

}