<?php
namespace TotalThemeCore\Vcex;

use Wpex_Post_Cards_Shortcode;

defined( 'ABSPATH' ) || exit;

class Post_Cards extends Wpex_Post_Cards_Shortcode {

	/**
	 * Output.
	 */
	protected $output = '';

	/**
	 * Unique element classname.
	 */
	protected $unique_classname = '';

	/**
	 * Associative array of shortcode attributes.
	 */
	protected $atts = [];

	/**
	 * Associative array of attributes used for ajax queries.
	 */
	protected $ajax_atts = [];

	/**
	 * Current query.
	 */
	protected $query = null;
	/**
	 * Pagination type.
	 */
	protected $pagination_type = null;

	/**
	 * Checks if ajax is being used.
	 */
	protected $has_ajax = false;

	/**
	 * Stores ajax action to prevent extra checks.
	 */
	protected $ajax_action = null;

	/**
	 * Constructor.
	 */
	public function __construct( $atts = [] ) {
		if ( ! vcex_maybe_display_shortcode( self::TAG, $atts )
			|| ! function_exists( 'vcex_build_wp_query' )
			|| ! function_exists( 'wpex_get_card' )
		) {
			return;
		}

		// Define shortcode output.
		$inner_output = '';

		// Store ajax atts before parsing.
		if ( ! empty( $atts ) ) {
			$this->ajax_atts = $atts;
		}

		// Check if currently loading items via load more.
		$is_doing_loadmore = $this->is_doing_ajax( 'load_more' );

		// Store some original atts before parsing.
		$og_entry_count = ( $is_doing_loadmore && ! empty( $atts['entry_count'] ) ) ? absint( $atts['entry_count'] ) : null;
		$running_count  = ( $is_doing_loadmore && ! empty( $atts['running_count'] ) ) ? absint( $atts['running_count'] ) : 0;

		// Parse shortcode atts.
		$this->atts = vcex_shortcode_atts( self::TAG, $atts, get_parent_class() );

		// Modify atts for custom queries.
		$this->maybe_modify_atts();

		// Core vars.
		$display_type       = $this->get_display_type();
		$grid_style         = $this->get_grid_style();
		$grid_columns       = $this->get_grid_columns();
		$grid_gap_class     = $this->get_grid_gap_class();
		$pagination_type    = $this->get_pagination_type();
		$grid_is_responsive = vcex_validate_att_boolean( 'grid_columns_responsive', $this->atts, true );

		// Featured card vars.
		$has_featured_card      = $this->has_featured_card();
		$featured_card_location = $this->get_featured_card_location();
		$is_featured_card_top   = $this->is_featured_card_top();
		$fc_bk                  = ! empty( $this->atts['featured_breakpoint'] ) ? $this->atts['featured_breakpoint'] : 'sm';

		// Get paged value from ajax atts.
		$this->atts['paged'] = ! empty( $this->ajax_atts['paged'] ) ? $this->ajax_atts['paged'] : null;

		// We can remove $atts from memory now.
		unset( $atts );

		// Parse featured card ID.
		if ( vcex_validate_att_boolean( 'featured_card', $this->atts ) ) {
			$this->atts['featured_post_id'] = $this->get_featured_post_id();
			$this->ajax_atts['featured_post_id'] = $this->atts['featured_post_id'];
		}

		// Set current entry count.
		$entry_count = $og_entry_count ?? 0;

		// Define card args.
		$card_args = $this->get_card_args();

		// Query posts.
		$this->query = vcex_build_wp_query( $this->atts, self::TAG );

		// Check if we have posts or a featured post.
		if ( ! $this->query->have_posts()
			&& empty( $this->atts['featured_post_id'] )
			&& ! $this->is_doing_ajax( 'filter' )
		) {
			$this->output = $this->no_posts_found_message();
			return;
		}

		$this->inline_style();

		/*-------------------------------------*/
		/* [ Inner Output Starts Here ]
		/*-------------------------------------*/
		$inner_class = [
			'wpex-post-cards-inner',
		];

		// Add flex styles to post cards inner.
		if ( $has_featured_card ) {
			if ( ! $is_featured_card_top ) {
				$inner_class[] = 'wpex-' . $fc_bk . '-flex';
			}
			if ( 'right' === $featured_card_location ) {
				$inner_class[] = 'wpex-' . $fc_bk . '-flex-row-reverse';
			}
		}

		$inner_output .= '<div class="' . esc_attr( implode( ' ', $inner_class ) ) . '">';

		/*-------------------------------------*/
		/* [ Featured Card ]
		/*-------------------------------------*/
		if ( $has_featured_card ) {

			$fc_width = apply_filters( 'wpex_post_cards_featured_width', 50 );
			$fc_width = ! empty( $this->atts['featured_width'] ) ? $this->atts['featured_width'] : $fc_width;

			$featured_card_classes = [
				'wpex-post-cards-featured',
			];

			// Featured card flex classes.
			if ( ! $is_featured_card_top ) {
				$featured_card_classes[] = 'wpex-' . $fc_bk . '-w-' . trim( absint( $fc_width ) );
				$featured_card_classes[] = 'wpex-' . $fc_bk . '-flex-shrink-0';
			}

			// Featured card bottom margin.
			if ( empty( $this->atts['featured_divider'] ) || ! $is_featured_card_top ) {
				$fc_margin = $this->atts['featured_margin'] ? absint( $this->atts['featured_margin'] ) : 30;
				$featured_card_classes[] = 'wpex-mb-' . $fc_margin;
			}

			// Featured card side margin.
			switch ( $featured_card_location ) {
				case 'left':
					$featured_card_classes[] = 'wpex-' . $fc_bk . '-mb-0';
					$featured_card_classes[] = 'wpex-' . $fc_bk . '-mr-' . $fc_margin;
					break;
				case 'right':
					$featured_card_classes[] = 'wpex-' . $fc_bk . '-mb-0';
					$featured_card_classes[] = 'wpex-' . $fc_bk . '-ml-' . $fc_margin;
					break;
			}

			if ( 'woocommerce' === $this->atts['card_style'] ) {
				$featured_card_classes[] = 'products';
			}

			// Display featured card.
			$inner_output .= '<div class="' . esc_attr( implode( ' ', $featured_card_classes ) ) . '">';

				if ( ! empty( $this->atts['featured_post_id'] ) ) {
					$featured_post_id = $this->atts['featured_post_id'];
					global $post;
					$post = get_post( $this->atts['featured_post_id'] );
					$inner_output .= wpex_get_card( $this->get_featured_card_args( $featured_post_id ) );
					wp_reset_postdata();
				} else {
					$count=0;
					while ( $this->query->have_posts() ) :
						$count++;
						if ( 2 === $count ) {
							break;
						}

						$this->query->the_post();

						$featured_post_id = get_the_ID();

						$inner_output .= wpex_get_card( $this->get_featured_card_args( $featured_post_id ) );

					endwhile;
				}

			$inner_output .= '</div>';

			if ( empty( $this->atts['featured_post_id'] ) ) {
				wp_reset_postdata();
				$this->query->rewind_posts();
			}

			if ( ! empty( $this->atts['featured_divider'] ) && $is_featured_card_top ) {
				$inner_output .= $this->featured_divider();
			}

		}

		/*-------------------------------------*/
		/* [ Entries start here ]
		/*-------------------------------------*/
		if ( $this->query->have_posts() ) {

			if ( $has_featured_card && ! $is_featured_card_top ) {
				$inner_output .= '<div class="wpex-post-cards-aside wpex-min-w-0 wpex-' . $fc_bk . '-flex-grow">';
			}

			// Before loop hook.
			ob_start();
				do_action( 'wpex_hook_post_cards_loop_before', $this->atts, $this->query );
				$inner_output .= ob_get_clean();

			// Get loop classes
			$items_wrap_class = [
				'wpex-post-cards-loop',
			];

			switch ( $display_type ) :

				case 'carousel':

					vcex_enqueue_carousel_scripts();

					// All carousels need a unique classname.
					if ( empty ( $this->unique_classname ) ) {
						$this->unique_classname = vcex_element_unique_classname();
					}

					// Get carousel settings.
					$carousel_settings = vcex_get_carousel_settings( $this->atts, self::TAG, false );
					$carousel_css = vcex_get_carousel_inline_css( $this->unique_classname . ' .wpex-posts-card-carousel', $carousel_settings );

					$items_data['data-wpex-carousel'] = vcex_carousel_settings_to_json( $carousel_settings );
					$items_wrap_class[] = 'wpex-posts-card-carousel';
					$items_wrap_class[] = 'wpex-carousel';
					$items_wrap_class[] = 'owl-carousel';

					if ( $carousel_css ) {
						$items_wrap_class[] = 'wpex-carousel--render-onload';
						$this->output .= $carousel_css;
					}

					// Flex carousel.
					if ( empty( $this->atts['auto_height'] ) || 'false' === $this->atts['auto_height'] ) {
						$items_wrap_class[] = 'wpex-carousel--flex';
					}

					// No margins.
					if ( array_key_exists( 'items_margin', $this->atts ) && empty( absint( $this->atts['items_margin'] ) ) ) {
						$items_wrap_class[] = 'wpex-carousel--no-margins';
					}

					// Arrow style.
					$arrows_style = ! empty( $this->atts['arrows_style'] ) ? $this->atts['arrows_style'] : 'default';
					$items_wrap_class[] = 'arrwstyle-' . sanitize_html_class( $arrows_style );

					// Arrow position.
					if ( ! empty( $this->atts['arrows_position'] ) && 'default' != $this->atts['arrows_position'] ) {
						$items_wrap_class[] = 'arrwpos-' . sanitize_html_class( $this->atts['arrows_position'] );
					}

					break;

				case 'list':
					$items_wrap_class[] = 'wpex-post-cards-list';
					$items_wrap_class[] = 'wpex-grid wpex-grid-cols-1'; // @note needs the grid class to prevent blowout.
					if ( $grid_gap_class ) {
						$items_wrap_class[] = $grid_gap_class;
					}
					if ( vcex_validate_att_boolean( 'alternate_flex_direction', $this->atts ) ) {
						$items_wrap_class[] = 'wpex-post-cards-list--alternate-flex-direction';
					}
					if ( vcex_validate_att_boolean( 'list_divider_remove_last', $this->atts ) ) {
						$items_wrap_class[] = 'wpex-last-divider-none';
					}
					break;

				case 'flex':
					$items_wrap_class[] = 'wpex-post-cards-flex wpex-flex';
					$flex_bk = ! empty( $this->atts['flex_breakpoint'] ) ? $this->atts['flex_breakpoint'] : '';
					if ( $flex_bk && 'false' !== $flex_bk ) {
						$items_wrap_class[] = 'wpex-flex-col';
						$items_wrap_class[] = 'wpex-' . sanitize_html_class( $flex_bk ) . '-flex-row';
					}
					$items_wrap_class[] = 'wpex-overflow-x-auto';
					if ( $grid_gap_class ) {
						$items_wrap_class[] = $grid_gap_class;
					}
					if ( vcex_validate_att_boolean( 'hide_scrollbar', $this->atts ) ) {
						$items_wrap_class[] = 'wpex-hide-scrollbar';
					}
					$snap_type = ! empty( $this->atts['flex_scroll_snap_type'] ) ? $this->atts['flex_scroll_snap_type'] : 'proximity';
					if ( 'proximity' === $snap_type || 'mandatory' === $snap_type ) {
						$has_scroll_snap = true;
						$items_wrap_class[] = 'wpex-snap-x';
						$items_wrap_class[] = 'wpex-snap-' . $snap_type;
					}
					break;

				case 'grid':
				default:

					if ( 'css_grid' === $grid_style ) {
						$items_wrap_class[] = 'wpex-post-cards-grid wpex-grid';
						if ( $grid_is_responsive && ! empty( $this->atts['grid_columns_responsive_settings'] ) ) {
							$r_grid_columns = vcex_parse_multi_attribute( $this->atts['grid_columns_responsive_settings'] );
							if ( $r_grid_columns && is_array( $r_grid_columns ) ) {
								$r_grid_columns['d'] = $grid_columns;
								$grid_columns = $r_grid_columns;
							}
						}
						if ( $grid_is_responsive && function_exists( 'wpex_grid_columns_class' ) ) {
							$items_wrap_class[] = wpex_grid_columns_class( $grid_columns );
						} else {
							$items_wrap_class[] = 'wpex-grid-cols-' . sanitize_html_class( $grid_columns );
						}
					} else {
						$items_wrap_class[] = 'wpex-post-cards-grid';
						$items_wrap_class[] = 'wpex-row';
						$items_wrap_class[] = 'wpex-clr';
					}

					if ( 'masonry' === $grid_style ) {
						$items_wrap_class[] = 'wpex-masonry-grid';
						if ( function_exists( 'wpex_enqueue_masonry_scripts' ) ) {
							wpex_enqueue_masonry_scripts(); // uses theme masonry scripts.
						}
					}

					if ( $grid_gap_class ) {
						$items_wrap_class[] = $grid_gap_class;
					}

					break;

			endswitch; // end display_type switch

			if ( 'woocommerce' === $this->atts['card_style'] ) {
				$items_wrap_class[] = 'products';
			}

			// Opens items wrap (wpex-post-cards-loop)
			$inner_output .= '<div class="' . esc_attr( implode( ' ', $items_wrap_class ) ) . '"';

				// Add grid data attributes.
				if ( ! empty( $items_data ) ) {
					foreach ( $items_data as $key => $value ) {
						$inner_output .= ' ' . $key ."='" . esc_attr( $value ) . "'";
					}
				}

				// Inner Items CSS
				// @todo move to Shortcode CSS somehow since it needs to target multiple selectors.
				$grid_css_args = [];
				if ( ! $grid_gap_class ) {
					switch ( $display_type ) {
						case 'flex':
						case 'grid':
							if ( ! empty( $this->atts['grid_spacing'] ) ) {
								if ( 'css_grid' === $grid_style || 'flex' === $display_type ) {
									$grid_css_args['gap'] = $this->atts['grid_spacing'];
								} else {
									$grid_css_args['--wpex-row-gap'] = $this->atts['grid_spacing'];
								}
							}
							break;
					}
				}
				if ( $grid_css_args ) {
					$inner_output .= vcex_inline_style( $grid_css_args );
				}

				$inner_output .= '>';

				// Add first divider if enabled.
				if ( 'list' === $display_type
					&& ! vcex_validate_att_boolean( 'list_divider_remove_first', $this->atts, true )
					&& ! $is_doing_loadmore
				) {
					$inner_output .= $this->list_divider( $this->atts );
				}

				// The Loop
				while ( $this->query->have_posts() ) :

					$this->query->the_post(); // !!! Important !!!

					$post_id = get_the_ID();
					$post_type = get_post_type( $post_id );
					$card_args['post_id'] = $post_id;

					if ( ! empty( $featured_post_id )
						&& empty( $this->atts['featured_post_id'] )
						&& $post_id === $featured_post_id
					) {
						continue;
					}

					$entry_count++;

					$running_count++;
					set_query_var( 'wpex_loop_running_count', absint( $running_count ) );

					$item_class = [
						'wpex-post-cards-entry',
						'post-' . sanitize_html_class( $post_id ),
						'type-' . sanitize_html_class( $post_type ),
					];

					switch ( $display_type ) :
						case 'carousel':
							$item_class[] = 'wpex-carousel-slide';
							break;
						case 'list':
							// No classes for list style.
							break;
						case 'flex':
						case 'grid':
						default:

							if ( 'flex' === $display_type ) {
								$item_class[] = 'wpex-flex';
								$item_class[] = 'wpex-flex-col';
								$item_class[] = 'wpex-max-w-100';
								if ( ! empty( $this->atts['flex_basis'] ) ) {
									$item_class[] = 'wpex-flex-shrink-0';
								} else {
									$item_class[] = 'wpex-flex-grow';
								}
								if ( isset( $has_scroll_snap ) && true === $has_scroll_snap ) {
									$item_class[] = 'wpex-snap-start';
								}
							}

							// Modern CSS grids.
							elseif ( 'css_grid' === $grid_style ) {
								$item_class[] = 'wpex-flex';
								$item_class[] = 'wpex-flex-col';
								$item_class[] = 'wpex-flex-grow';
							}

							// Old school grids.
							else {
								if ( $grid_is_responsive ) {
									$item_class[] = 'col';
								} else {
									$item_class[] = 'nr-col';
								}

								$item_class[] = 'col-' . sanitize_html_class( $entry_count );

								if ( $grid_columns ) {
									$item_class[] = 'span_1_of_' . sanitize_html_class( $grid_columns );
								}

								if ( $grid_is_responsive ) {
									$rs = vcex_parse_multi_attribute( $this->atts['grid_columns_responsive_settings'] );
									foreach ( $rs as $key => $val ) {
										if ( $val ) {
											$item_class[] = 'span_1_of_' . sanitize_html_class( $val ) . '_' . sanitize_html_class( $key );
										}
									}
								}
							}

							if ( 'masonry' === $grid_style ) {
								$item_class[] = 'wpex-masonry-col';
							}

							break;

					endswitch;

					if ( function_exists( 'vcex_get_post_term_classes' ) ) {
						$terms = vcex_get_post_term_classes();
						if ( $terms && is_array( $terms ) ) {
							foreach ( $terms as $term_name ) {
								$item_class[] = $term_name;
							}
						}
					}

					/**
					 * Filters the Post Cards shortcode entry classes.
					 *
					 * @param array $class
					 * @param array $shortcode_atts
					 */
					$item_class = (array) apply_filters( 'wpex_post_cards_entry_class', $item_class, $this->atts );

					// Begin entry output.
					$inner_output .= '<div class="' . esc_attr( implode( ' ', array_unique( $item_class ) ) ) . '">';

						$inner_output .= wpex_get_card( $card_args );

					$inner_output .= '</div>';

					// List Divider.
					if ( 'list' === $display_type && ! empty( $this->atts['list_divider'] ) ) {
						$inner_output .= $this->list_divider( $this->atts );
					}

					// Reset entry count.
					if ( 'grid' === $display_type
						&& 'fit_rows' === $grid_style
						&& $entry_count === $grid_columns
					) {
						$entry_count = 0;
					}

				endwhile;

			// Update ajax vars after loop.
			$this->ajax_atts['entry_count'] = $entry_count;
			$this->ajax_atts['running_count'] = $running_count;

			// Reset post data.
			wp_reset_postdata();

			// Remove running count.
			set_query_var( 'wpex_loop_running_count', null );

			// Close element that holds main (not featured) posts - wpex-post-cards-loop
			$inner_output .= '</div>';

			// After loop hook.
			ob_start();
				do_action( 'wpex_hook_post_cards_loop_after', $this->atts, $this->query );
			$inner_output .= ob_get_clean();

			// Pagination.
			if ( 'numbered' !== $pagination_type && 'numbered_ajax' !== $pagination_type ) {
				$pagination_added = true;
				$inner_output .= $this->get_pagination();
			}

			// Close featured aside wrap.
			if ( $has_featured_card && ! $is_featured_card_top ) {
				$inner_output .= '</div>';
			}

		} else {

			if ( $this->is_doing_ajax( 'filter' ) ) {
				$inner_output .= $this->no_posts_found_message();
			}

		} // end has posts check

		// Close post cards inner.
		$inner_output .= '</div>';

		// Outer pagination.
		if ( ! isset( $pagination_added ) ) {
			$inner_output .= $this->get_pagination();
		}

		// Ajax Loader (must run after get_pagination)
		if ( ! empty( $this->atts['unique_id'] ) || true === $this->has_ajax ) {
			$inner_output .= Ajax::instance()->get_ajax_loader();
		}

		/*-------------------------------------*/
		/* [ Put inner_output inside wrap. ]
		/*-------------------------------------*/
		$this->output .= '<div';

			// Wrap id attribute.
			if ( ! empty( $this->atts['unique_id'] ) ) {
				$this->output .= ' id="' . esc_attr( $this->atts['unique_id'] ) . '"';
			}

			// Wrap class attribute.
			$this->output .= ' class="' . esc_attr( implode( ' ', $this->get_wrap_classes( $has_featured_card ) ) ) . '"';

			// Wrap data attributes.
			if ( ! empty( $this->atts['unique_id'] ) || true === $this->has_ajax ) {
				$this->output .= ' data-vcex-class="' . get_parent_class() . '"';
				$this->output .= ' data-vcex-atts="' . $this->get_json_data() . '"';
				$this->output .= ' data-vcex-max-pages="' . esc_attr( $this->query->max_num_pages ?? 0 )  .'"';
				$this->output .= ' data-vcex-current-page="' . esc_attr( get_query_var( 'paged' ) ?: 1 ) . '"';
			}

			if ( 'numbered_ajax' === $pagination_type ) {
				$this->output .= ' data-vcex-pagination="numbered_ajax"';
			}

			$this->output .= '>';

			$this->output .= $inner_output;

		$this->output .= '</div>'; // close wrap
	}

	/**
	 * Adds inline style to output.
	 */
	protected function inline_style() {
		$css = '';
		$shortcode_css = new Shortcode_CSS( get_parent_class(), $this->atts );
		$shortcode_style = $shortcode_css->render_style( false );
		if ( $shortcode_style && ! empty( $shortcode_css->unique_classname ) ) {
			$css .= $shortcode_style;
			$unique_classname = $shortcode_css->unique_classname;
		}

		// Flex basis needs to be calculated differently.
		$css_xtra = '';
		$display_type = $this->get_display_type();
		if ( 'flex' === $display_type && ! empty( $this->atts['flex_basis'] ) ) {
			$flex_bk = $this->get_breakpoint_px( $this->atts['flex_breakpoint'] ?? null );
			$flex_basis = '{{class}} .wpex-post-cards-entry{flex-basis:' . $this->parse_flex_basis( $this->atts['flex_basis'] ) . '}';
			if ( $flex_bk ) {
				$css_xtra .= '@media only screen and (min-width: ' . $flex_bk . ') {' . $flex_basis . '}';
			} else {
				$css_xtra .= $flex_basis;
			}
			if ( $css_xtra ) {
				$unique_classname = $unique_classname ?? vcex_element_unique_classname();
				$css .= str_replace( '{{class}}', '.' . $unique_classname, $css_xtra );
			}
		}

		if ( $css ) {
			$this->unique_classname = $unique_classname;
			$this->output .= '<style>' . $css . '</style>';
		}
	}

	/**
	 * Modifies atts.
	 */
	protected function maybe_modify_atts() {
		$query_type = $this->get_query_type();
		if ( in_array( $query_type, [ 'post_gallery', 'attachments' ] ) ) {
			if ( ! $this->is_doing_ajax( 'any' ) ) {
				switch ( $query_type ) {
					case 'post_gallery':
						if ( function_exists( 'wpex_get_gallery_ids' ) ) {
							$gallery_ids = wpex_get_gallery_ids();
						}
						break;
					case 'attachments':
						if ( ! empty( $this->atts['attachments'] ) ) {
							if ( is_string( $this->atts['attachments'] ) ) {
								$gallery_ids = explode( ',', $this->atts['attachments'] );
							}
							if ( is_array( $this->atts['attachments'] ) ) {
								if ( $this->is_elementor_widget() ) {
									$gallery_ids = array_column( $this->atts['attachments'], 'id' );
								}
							}
						}
						break;
				}
				$this->atts['query_type'] = 'custom';
				$this->atts['custom_query_args'] = [
					'post_type'      => 'attachment',
					'post_status'    => 'any',
					'post__in'       => $gallery_ids ?? [ 0 ],
					'orderby'        => 'post__in',
					'posts_per_page' => ! empty( $this->atts['posts_per_page'] ) ? sanitize_text_field( $this->atts['posts_per_page'] ) : '12',
				];
				// Important we need to reset ajax atts.
				$this->ajax_atts['query_type'] = $this->atts['query_type'];
				$this->ajax_atts['custom_query_args'] = $this->atts['custom_query_args'];
				// Remove featured post if enabled.
			}
		}
	}

	/**
	 * Parses the flex basis and returns correct value.
	 */
	protected function parse_flex_basis( $basis = '' ) {
		switch ( $basis ) {
			case '1':
				return '100%';
				break;
			case '2':
			case '3':
			case '4':
				$basis = absint( $basis );
				$gap_count = $basis - 1;
				$gap = ! empty( $this->atts['grid_spacing'] ) ? wp_strip_all_tags( $this->atts['grid_spacing'] ) : '20px';
				if ( $gap && 'none' !== $gap ) {
					if ( is_numeric( $gap ) ) {
						$gap = $gap . 'px';
					}
					return 'calc((100% / ' . $basis . ') - ((' . $gap . ' * ' . $gap_count . ') / ' . $basis . '))';
				} else {
					return 'calc(100% / ' . $basis . ')';
				}
				break;
			default:
				return sanitize_text_field( $basis );
				break;
		}
	}

	/**
	 * Returns a breakpoint in pixels based on selected option.
	 */
	protected function get_breakpoint_px( $breakpoint = '' ) {
		switch ( $breakpoint ) {
			case 'xl':
				return '1280px';
				break;
			case 'lg':
				return '1024px';
				break;
			case 'md':
				return '768px';
				break;
			case 'sm':
				return '640px';
				break;
		}
	}

	/**
	 * Return array of wrap classes.
	 */
	protected function get_wrap_classes( $has_featured_card = false ) {
		$classes = [
			'wpex-post-cards',
			'wpex-post-cards-' . sanitize_html_class( $this->atts['card_style'] ),
		];

		if ( $has_featured_card ) {
			$classes[] = 'wpex-post-cards-has-featured'; // @todo rename to use BEM
		}

		if ( 'woocommerce' === $this->atts['card_style'] ) {
			$classes[] = 'woocommerce';
		}

		if ( ! empty( $this->atts['bottom_margin'] ) ) {
			$classes[] = vcex_sanitize_margin_class( $this->atts['bottom_margin'], 'wpex-mb-' );
		}

		if ( ! empty( $this->atts['el_class'] ) ) {
			$classes[] = vcex_get_extra_class( $this->atts['el_class'] );
		}

		if ( ! empty( $this->atts['css_animation'] ) && vcex_validate_att_boolean( 'css_animation_sequential', $this->atts ) ) {
			$classes[] = 'wpb-animate-in-sequence';
		}

		if ( ! empty( $this->unique_classname ) ) {
			$classes[] = $this->unique_classname;
		}

		$classes[] = 'wpex-relative';

		return $classes;
	}

	/**
	 * Return json data used for ajax functions.
	 */
	protected function get_json_data() {
		if ( $this->is_auto_query() ) {
			if ( empty( $this->ajax_atts['query_vars'] ) ) {
				$this->ajax_atts['query_vars'] = wp_json_encode( $this->query->query_vars );
			}
			$this->ajax_atts['query_vars'] = $this->ajax_atts['query_vars'];
		}
		unset( $this->ajax_atts['ajax_action'] ); // not needed anymore
		unset( $this->ajax_atts['ajax_filter'] ); // not needed anymore
		return esc_attr( wp_json_encode( $this->ajax_atts, false ) );
	}

	/**
	 * Return card args based on shortcode atts.
	 */
	protected function get_card_args() {
		$args = [
			'style' => $this->atts['card_style'],
		];

		if ( ! empty( $this->atts['template_id'] ) ) {
			$args['template_id'] = $this->atts['template_id'];
		}

		if ( ! empty( $this->atts['date_format'] ) ) {
			$args['date_format'] = $this->atts['date_format'];
		}

		if ( ! empty( $this->atts['display_type'] ) ) {
			$args['display_type'] = $this->atts['display_type'];
		}

		if ( ! empty( $this->atts['link_type'] ) ) {
			$args['link_type'] = $this->atts['link_type'];
		}

		if ( ! empty( $this->atts['modal_title'] ) ) {
			$args['modal_title'] = $this->atts['modal_title'];
		}

		if ( ! empty( $this->atts['modal_template'] ) ) {
			$args['modal_template'] = $this->atts['modal_template'];
		}

		if ( ! empty( $this->atts['link_target'] ) ) {
			$args['link_target'] = $this->atts['link_target'];
		}

		if ( ! empty( $this->atts['link_rel'] ) ) {
			$args['link_rel'] = $this->atts['link_rel'];
		}

		if ( ! empty( $this->atts['title_font_size'] ) ) {
			$args['title_font_size'] = $this->atts['title_font_size'];
		}

		if ( ! empty( $this->atts['title_tag'] ) ) {
			$args['title_tag'] = $this->atts['title_tag'];
		}

		if ( ! empty( $this->atts['css_animation'] ) ) {
			$args['css_animation'] = $this->atts['css_animation'];
		}

		if ( isset( $this->atts['more_link_text'] ) && '' !== $this->atts['more_link_text'] ) {
			$args['more_link_text'] = $this->atts['more_link_text']; // allows "0" for disabling.
		}

		if ( ! empty( $this->atts['media_width'] ) ) {
			$args['media_width'] = $this->atts['media_width'];
		}

		if ( ! empty( $this->atts['media_breakpoint'] ) ) {
			$args['breakpoint'] = $this->atts['media_breakpoint'];
		}

		if ( empty( $this->atts['thumbnail_size'] ) || 'wpex_custom' === $this->atts['thumbnail_size'] ) {
			$args['thumbnail_size'] = [
				$this->atts['thumbnail_width'],
				$this->atts['thumbnail_height'],
				$this->atts['thumbnail_crop'],
			];
		} else {
			$args['thumbnail_size'] = $this->atts['thumbnail_size'];
		}

		if ( ! empty( $this->atts['thumbnail_overlay_style'] ) ) {
			$args['thumbnail_overlay_style'] = $this->atts['thumbnail_overlay_style'];
		}

		if ( ! empty( $this->atts['thumbnail_overlay_button_text'] ) ) {
			$args['thumbnail_overlay_button_text'] = $this->atts['thumbnail_overlay_button_text'];
		}

		if ( ! empty( $this->atts['thumbnail_hover'] ) ) {
			$args['thumbnail_hover'] = $this->atts['thumbnail_hover'];
		}

		if ( ! empty( $this->atts['thumbnail_filter'] ) ) {
			$args['thumbnail_filter'] = $this->atts['thumbnail_filter'];
		}

		if ( ! empty( $this->atts['media_el_class'] ) ) {
			$args['media_el_class'] = vcex_get_extra_class( $this->atts['media_el_class'] );
		}

		if ( ! empty( $this->atts['card_el_class'] ) ) {
			$args['el_class'] = vcex_get_extra_class( $this->atts['card_el_class'] );
		}

		if ( isset( $this->atts['excerpt_length'] ) && '' !== $this->atts['excerpt_length'] ) {
			$args['excerpt_length'] = $this->atts['excerpt_length'];
		}

		if ( ! empty( $this->atts['alternate_flex_direction'] ) ) {
			$args['alternate_flex_direction'] = $this->atts['alternate_flex_direction'];
		}

		$allowed_media = $this->get_allowed_media();

		if ( ! is_null( $allowed_media ) ) {
			$args['allowed_media'] = $allowed_media;
		}

		return $args;
	}

	/**
	 * Check if loadmore is enabled.
	 */
	protected function has_loadmore() {
		if ( ( ! empty( $this->atts['pagination'] ) && in_array( $this->atts['pagination'], [ 'loadmore', 'infinite_scroll' ] ) ) || vcex_validate_att_boolean( 'pagination_loadmore', $this->atts ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Check if featured card is enabled.
	 */
	protected function has_featured_card() {
		if ( $this->is_doing_ajax( 'load_more' ) ) {
			return false;
		}

		// Check if the featured card is enabled.
		$check = vcex_validate_att_boolean( 'featured_card', $this->atts );

		// Do not show featured card on paginated pages.
		if ( ( ! vcex_validate_att_boolean( 'featured_show_on_paged', $this->atts ) || $this->has_loadmore() ) && is_paged() ) {
			$check = false;
		}

		/**
		 * Filters whether the featured card is enabled or not.
		 *
		 * @param bool $check
		 */
		$check = (bool) apply_filters( 'wpex_post_cards_has_featured_card', $check, $this->atts );

		return $check;
	}

	/**
	 * Get supported media.
	 */
	protected function get_allowed_media() {
		if ( $this->is_elementor_widget() ) {
			return null;
		}
		if ( array_key_exists( 'allowed_media', $this->atts ) ) {
			if ( $this->atts['allowed_media'] ) {
				if ( is_string( $this->atts['allowed_media'] ) ) {
					$this->atts['allowed_media'] = wp_parse_list( $this->atts['allowed_media'] );
				}
				foreach ( $this->atts['allowed_media'] as $k => $v ) {
					if ( ! in_array( $v, [ 'thumbnail', 'video' ] ) ) {
						unset( $this->atts['allowed_media'][$k] );
					}
				}
			}
			return $this->atts['allowed_media'];
		}
	}

	/**
	 * Check if the featured card should use a custom query.
	 */
	protected function has_featured_card_custom_query() {
		if ( $this->has_loadmore()
			|| ! vcex_validate_att_boolean( 'featured_show_on_paged', $this->atts )
		) {
			return true;
		}
		return false;
	}

	/**
	 * Get featured card ID.
	 */
	protected function get_featured_post_id() {
		$featured_post_id = 0;
		if ( ! empty( $this->atts['featured_post_id'] )
			&& ! $this->is_doing_ajax( [ 'pagination', 'filter' ] )
		) {
			$post_id = $this->atts['featured_post_id'];
			$post = get_post( $post_id );
			if ( $post && 'publish' === $post->post_status ) {
				$featured_post_id = wpex_parse_obj_id( $post_id, get_post_type( $post ) );
			}
		} elseif ( $this->has_featured_card_custom_query() ) {
			$query_args = $this->atts;
			$query_args['posts_per_page'] = 1;
			$query_posts = vcex_build_wp_query( $query_args );
			if ( $query_posts->have_posts() && ! empty( $query_posts->posts[0] ) ) {
				$featured_post_id = $query_posts->posts[0]->ID ?? 0;
			}
		}

		/**
		 * Filters the featured card ID.
		 *
		 * @param int $featured_post_id
		 * @param array $shortcode attributes
		 */
		$featured_post_id = (int) apply_filters( 'wpex_post_cards_featured_post_id', $featured_post_id, $this->atts );

		return $featured_post_id;
	}

	/**
	 * Featured card args.
	 *
	 * @todo can we optimize this to use a loop instead?
	 */
	protected function get_featured_card_args( $post_id ) {
		$featured_card_style = ! empty( $this->atts['featured_style'] ) ? $this->atts['featured_style'] : $this->atts['card_style'];

		$args = [
			'post_id'  => $post_id,
			'style'    => $featured_card_style,
			'featured' => true,
		];

		if ( ! empty( $this->atts['featured_template_id'] ) ) {
			$args['template_id'] = $this->atts['featured_template_id'];
		}

		if ( ! empty( $this->atts['date_format'] ) ) {
			$args['date_format'] = $this->atts['date_format'];
		}

		if ( ! empty( $this->atts['featured_title_font_size'] ) ) {
			$args['title_font_size'] = $this->atts['featured_title_font_size'];
		}

		if ( ! empty( $this->atts['featured_title_tag'] ) ) {
			$args['title_tag'] = $this->atts['featured_title_tag'];
		}

		if ( empty( $this->atts['featured_thumbnail_size'] ) || 'wpex_custom' === $this->atts['featured_thumbnail_size'] ) {
			$args['thumbnail_size'] = [
				$this->atts['featured_thumbnail_width'],
				$this->atts['featured_thumbnail_height'],
				$this->atts['featured_thumbnail_crop'],
			];
		} else {
			$args['thumbnail_size'] = $this->atts['featured_thumbnail_size'];
		}

		if ( isset( $this->atts['featured_more_link_text'] ) && '' !== $this->atts['featured_more_link_text'] ) {
			$args['more_link_text'] = $this->atts['featured_more_link_text']; // allows "0" for disabling.
		}

		if ( isset( $this->atts['featured_excerpt_length'] ) && '' !== $this->atts['featured_excerpt_length'] ) {
			$args['excerpt_length'] = $this->atts['featured_excerpt_length'];
		}

		if ( ! empty( $this->atts['thumbnail_overlay_style'] ) ) {
			$args['thumbnail_overlay_style'] = $this->atts['thumbnail_overlay_style'];
		}

		if ( ! empty( $this->atts['thumbnail_overlay_button_text'] ) ) {
			$args['thumbnail_overlay_button_text'] = $this->atts['thumbnail_overlay_button_text'];
		}

		if ( ! empty( $this->atts['thumbnail_hover'] ) ) {
			$args['thumbnail_hover'] = $this->atts['thumbnail_hover'];
		}

		if ( ! empty( $this->atts['thumbnail_filter'] ) ) {
			$args['thumbnail_filter'] = $this->atts['thumbnail_filter'];
		}

		if ( ! empty( $this->atts['media_el_class'] ) ) {
			$args['media_el_class'] = $this->atts['media_el_class'];
		}

		if ( ! empty( $this->atts['featured_el_class'] ) ) {
			$args['el_class'] = $this->atts['featured_el_class'];
		}

		if ( ! empty( $this->atts['modal_title'] ) ) {
			$args['modal_title'] = $this->atts['modal_title'];
		}

		if ( ! empty( $this->atts['modal_template'] ) ) {
			$args['modal_template'] = $this->atts['modal_template'];
		}

		if ( ! empty( $this->atts['link_type'] ) ) {
			$args['link_type'] = $this->atts['link_type'];
		}

		if ( ! empty( $this->atts['link_target'] ) ) {
			$args['link_target'] = $this->atts['link_target'];
		}

		if ( ! empty( $this->atts['link_rel'] ) ) {
			$args['link_rel'] = $this->atts['link_rel'];
		}

		if ( ! empty( $this->atts['featured_media_width'] ) ) {
			$args['media_width'] = $this->atts['featured_media_width'];
		}

		if ( ! empty( $this->atts['featured_media_breakpoint'] ) ) {
			$args['breakpoint'] = $this->atts['featured_media_breakpoint'];
		} elseif ( ! empty( $this->atts['media_breakpoint'] ) ) {
			$args['breakpoint'] = $this->atts['media_breakpoint'];
		}

		$allowed_media = $this->get_allowed_media();

		if ( ! is_null( $allowed_media ) ) {
			$args['allowed_media'] = $allowed_media;
		}

		/**
		 * Filters the wpex_post_card featured card args.
		 *
		 * @param array $args
		 * @param array $shortcode_attributes
		 */
		$args = (array) apply_filters( 'wpex_post_cards_featured_card_args', $args, $this->atts );

		return $args;
	}

	/**
	 * Get featured card location.
	 */
	protected function get_featured_card_location() {
		return $this->atts['featured_location'] ?? 'top';
	}

	/**
	 * Check if featured card is enabled.
	 */
	protected function is_featured_card_top() {
		$location = $this->get_featured_card_location();
		if ( 'left' === $location || 'right' === $location ) {
			return false;
		}
		return true;
	}

	/**
	 * Featured Card Divider
	 */
	protected function featured_divider() {
		$divider_class = [
			'wpex-post-cards-featured-card-divider',
			'wpex-divider',
			'wpex-divider-' . sanitize_html_class( $this->atts['featured_divider'] ),
		];

		if ( ! empty( $this->atts['featured_divider_size'] ) ) {
			$divider_size = absint( $this->atts['featured_divider_size'] );
			if ( 1 === $divider_size ) {
				$divider_class[] = 'wpex-border-b';
			} else {
				$divider_class[] = 'wpex-border-b-' . $divider_size;
			}
		}

		$spacing = ! empty( $this->atts['featured_divider_margin'] ) ? $this->atts['featured_divider_margin'] : 15;

		if ( ! empty( $this->atts['featured_margin'] ) ) {
			$divider_class[] = 'wpex-mt-' . sanitize_html_class( absint( $spacing ) );
			$divider_class[] = 'wpex-mb-' . sanitize_html_class( absint( $this->atts['featured_margin'] ) );
		} else {
			$divider_class[] = 'wpex-my-' . sanitize_html_class( absint( $spacing ) );
		}

		return '<div class="' . esc_attr( implode( ' ', $divider_class ) ) . '"></div>';
	}

	/**
	 * List Divider.
	 */
	protected function list_divider() {
		$divider_class = [
			'wpex-card-list-divider',
			'wpex-divider',
			'wpex-divider-' . sanitize_html_class( $this->atts['list_divider'] ),
		];

		$divider_class[] = 'wpex-my-0'; // remove default margin since we want to use gaps.

		if ( ! empty( $this->atts['list_divider_size'] ) ) {
			$divider_size = absint( $this->atts['list_divider_size'] );
			if ( 1 === $divider_size ) {
				$divider_class[] = 'wpex-border-b';
			} else {
				$divider_class[] = 'wpex-border-b-' . $divider_size;
			}
		}

		return '<div class="' . esc_attr( implode( ' ', $divider_class ) ) . '"></div>';
	}

	/**
	 * Get display type.
	 */
	protected function get_display_type() {
		return ! empty( $this->atts['display_type'] ) ? $this->atts['display_type'] : 'grid';
	}

	/**
	 * Get pagination type.
	 */
	protected function get_pagination_type() {
		if ( ! is_null( $this->pagination_type ) ) {
			return $this->pagination_type;
		}

		$pagination_type = $this->atts['pagination'] ?? '';
		$is_auto_query   = $this->is_auto_query();

		$allowed_choices = [
			'loadmore',
			'numbered',
			'numbered_ajax',
			'infinite_scroll',
		];

		// Allowed pagination styles.
		if ( ! in_array( $pagination_type, $allowed_choices ) ) {
			$pagination_type = '';
		}

		// Enable pagination for auto and custom queries.
		if ( ! $pagination_type && ( $is_auto_query || ( vcex_validate_att_boolean( 'custom_query', $this->atts ) && ! empty( $this->query->query['pagination'] ) ) ) ) {
			$pagination_type = 'numbered';
		}

		// relevanssi fix.
		if ( $pagination_type && $is_auto_query && function_exists( 'relevanssi_do_query' ) && is_search() ) {
			$pagination_type = 'numbered';
		}

		$this->pagination_type = $pagination_type;

		return $this->pagination_type;
	}

	/**
	 * Get grid style.
	 */
	protected function get_grid_style() {
		$allowed = [
			'css_grid',
			'masonry',
			'fit_rows',
		];
		if ( ! empty( $this->atts['grid_style'] ) && in_array( $this->atts['grid_style'], $allowed ) ) {
			return $this->atts['grid_style'];
		}
		return 'fit_rows';
	}

	/**
	 * Get grid style.
	 */
	protected function get_grid_columns() {
		return ! empty( $this->atts['grid_columns'] ) ? absint( $this->atts['grid_columns'] ) : 1;
	}

	/**
	 * Get grid gap class.
	 */
	protected function get_grid_gap_class() {
		$display_type = $this->get_display_type();
		if ( 'carousel' === $display_type ) {
			return;
		}
		$gap        = '';
		$grid_style = $this->get_grid_style();
		switch ( $display_type ) {
			case 'list':
				$gap = ! empty( $this->atts['list_spacing'] ) ? $this->atts['list_spacing'] : '15';
				break;
			case 'flex':
			case 'grid':
				$default = '';
				if ( 'css_grid' === $grid_style || 'flex' === $display_type ) {
					$default = '20';  // css_grid needs a default gap.
				}
				$gap = ! empty( $this->atts['grid_spacing'] ) ? $this->atts['grid_spacing'] : $default;
				break;
		}
		if ( $gap ) {
			if ( 'list' === $display_type
				|| 'flex' === $display_type
				|| ( 'grid' === $display_type && 'css_grid' === $grid_style )
			) {
				$use_utl_class = true;
			} else {
				$use_utl_class = false;
			}
			if ( 'none' === $gap ) {
				if ( $use_utl_class ) {
					return 'wpex-gap-0';
				} else {
					return 'gap-none';
				}
			}
			if ( function_exists( 'wpex_column_gaps' )
				&& array_key_exists( str_replace( 'px', '', $gap ), wpex_column_gaps() )
			) {
				if ( $use_utl_class ) {
					return 'wpex-gap-' . absint( $gap );
				} else {
					return 'gap-' . absint( $gap );
				}
			}
		}
	}

	/**
	 * Returns pagination output.
	 */
	protected function get_pagination() {
		$max_pages = absint( $this->query->max_num_pages ?? 1 );
		if ( ! $max_pages || 1 === $max_pages ) {
			return;
		}
		$html = '';
		$display_type = $this->get_display_type();
		if ( 'grid' === $display_type || 'list' === $display_type || 'flex' === $display_type ) {
			$pagination_type = $this->get_pagination_type();
			switch ( $pagination_type ) {
				case 'loadmore';
				case 'infinite_scroll';
					$this->has_ajax = true;
					Ajax::instance()->enqueue_scripts( get_parent_class(), $this->atts );
					$infinite_scroll = ( 'infinite_scroll' === $pagination_type ) ? true : false;
					$html .= Ajax::instance()->get_loadmore_button( [
						'shortcode_tag'   => self::TAG,
						'shortcode_atts'  => $this->ajax_atts,
						'query'           => $this->query,
						'infinite_scroll' => $infinite_scroll,
						'loadmore_text'   => ! empty( $this->atts['loadmore_text'] ) ? $this->atts['loadmore_text'] : '',
					] );
					break;
				case 'numbered_ajax':
					$this->has_ajax = true;
					Ajax::instance()->enqueue_scripts( get_parent_class(), $this->atts );
					$html .= vcex_pagination( $this->query, false );
					break;
				case 'numbered';
					$html .= vcex_pagination( $this->query, false );
					break;
			}
		}
		if ( $html ) {
			$html = '<div class="wpex-post-cards-pagination wpex-mt-30 wpex-first-mt-0">' . $html .'</div>';
		}
		return $html;
	}

	/**
	 * Returns the query type.
	 */
	protected function get_query_type() {
		return $this->atts['query_type'] ?? '';
	}

	/**
	 * Check if we are showing an auto query.
	 */
	protected function is_auto_query() {
		if ( 'auto' === $this->get_query_type() ) {
			return true;
		} else {
			return vcex_validate_att_boolean( 'auto_query', $this->atts );
		}
	}

	/**
	 * Check if displaying within an elementor widget.
	 */
	protected function is_elementor_widget() {
		return vcex_validate_att_boolean( 'is_elementor_widget', $this->atts );
	}

	/**
	 * Get current AJAX action.
	 */
	protected function get_ajax_action() {
		if ( ! is_null( $this->ajax_action ) ) {
			return $this->ajax_action;
		}
		$this->ajax_action = false;
		$action = $this->ajax_atts['ajax_action'] ?? null;
		if ( $action && wp_doing_ajax() && isset( $_REQUEST['action'] ) && Ajax::ACTION === $_REQUEST['action'] ) {
			$this->ajax_action = $action;
		}
		return $this->ajax_action;
	}

	/**
	 * Check if we are currently doing a specific ajax action.
	 */
	protected function is_doing_ajax( $action = '' ) {
		if ( ! is_array( $action ) ) {
			$action = [ $action ];
		}
		if ( 'any' === $action ) {
			return (bool) $this->get_ajax_action();
		}
		if ( in_array( $this->get_ajax_action(), $action ) ) {
			return true;
		}
	}

	/**
	 * Check if we are currently doing a specific ajax action.
	 */
	protected function no_posts_found_message() {
		if ( $this->is_doing_ajax( 'load_more' ) ) {
			return;
		}

		if ( ! empty( $this->atts['no_posts_found_message'] ) ) {
			$custom_message_safe = sanitize_text_field( $this->atts['no_posts_found_message'] );
			return '<div class="vcex-no-posts-found">' . do_shortcode( $custom_message_safe ) . '</div>';
		}

		$message = null;
		$check   = false;

		if ( vcex_vc_is_inline() || $this->is_auto_query() || $this->is_doing_ajax( 'filter' ) ) {
			$check = true;
		}

		$check = (bool) apply_filters( 'vcex_has_no_posts_found_message', $check, $this->atts );

		if ( $check ) {
			$message = '<div class="vcex-no-posts-found">' . esc_html__( 'Nothing found.', 'total-theme-core' ) . '</div>';
		}

		/**
		 * Apply filters to the no posts found message.
		 *
		 * @param string $message
		 * @param array $shortcode_atts
		 */
		$message = (string) apply_filters( 'vcex_no_posts_found_message', $message, $this->atts );

		return $message;
	}

	/**
	 * Outputs the html.
	 */
	public function render() {
		echo $this->output; // @codingStandardsIgnoreLine
	}

	/**
	 * Returns the html.
	 */
	public function get_output() {
		return $this->output;
	}

	/**
	 * Returns the query.
	 */
	public function get_query() {
		return $this->query;
	}

	/**
	 * Returns the parsed atts.
	 */
	public function get_atts() {
		return $this->atts;
	}

}