<?php
namespace TotalThemeCore\Vcex\Carousel;

use TotalThemeCore\Vcex\Scripts;
use TotalTheme\Theme_Icons;

defined( 'ABSPATH' ) || exit;

/**
 * Core Carousel methods.
 *
 * @package TotalThemeCore
 * @version 1.5
 */

final class Core {

	/**
	 * Returns list of style dependencies.
	 */
	public static function get_style_depends() {
		return [
			'wpex-owl-carousel',
		];
	}

	/**
	 * Returns list of script dependencies.
	 */
	public static function get_script_depends() {
		return [
			'wpex-owl-carousel',
			'vcex-carousels',
		];
	}

	/**
	 * Registers the carousel scripts.
	 */
	public static function register_scripts() {
		$js_extension = Scripts::instance()->get_js_extension();

		wp_register_style(
			'wpex-owl-carousel',
			vcex_asset_url( 'css/wpex-owl-carousel.css' ),
			[],
			'2.3.4'
		);

		wp_register_script(
			'wpex-owl-carousel',
			vcex_asset_url( 'js/lib/wpex-owl-carousel' . $js_extension ),
			[ 'jquery' ],
			TTC_VERSION,
			true
		);

		wp_register_script(
			'vcex-carousels',
			vcex_asset_url( 'js/vcex-carousels' . $js_extension ),
			[ 'jquery', 'wpex-owl-carousel', 'imagesloaded' ],
			TTC_VERSION,
			true
		);

		wp_localize_script( 'vcex-carousels', 'vcex_carousels_params', self::get_default_settings() );
	}

	/**
	 * Enqueues the carousel scripts.
	 */
	public static function enqueue_scripts() {
		foreach ( self::get_style_depends() as $style ) {
			wp_enqueue_style( $style );
		}

		foreach ( self::get_script_depends() as $script ) {
			wp_enqueue_script( $script );
		}

		// Theme icons are required currently - @todo update to use SVG's
		if ( is_callable( 'TotalTheme\\Theme_Icons::enqueue_font_style' ) ) {
			Theme_Icons::enqueue_font_style();
		}
	}

	/**
	 * Returns shortcode params.
	 */
	public static function get_shortcode_params( $dependency = [], $group = '' ) {
		$carousel_defaults = self::get_default_settings();
		$params = array(
			array(
				'type'       => 'vcex_subheading',
				'param_name' => 'vcex_subheading__carousel',
				'text'       => esc_html__( 'Carousel Settings', 'total-theme-core' ),
			),
			array(
				'type' => 'vcex_ofswitch',
				'heading' => esc_html__( 'Arrows', 'total-theme-core' ),
				'param_name' => 'arrows',
				'std' => 'true',
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_select',
				'choices' => 'carousel_arrow_styles',
				'heading' => esc_html__( 'Arrows Style', 'total-theme-core' ),
				'param_name' => 'arrows_style',
				'dependency' => array( 'element' => 'arrows', 'value' => 'true' ),
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_select',
				'choices' => 'carousel_arrow_positions',
				'heading' => esc_html__( 'Arrows Position', 'total-theme-core' ),
				'param_name' => 'arrows_position',
				'dependency' => array( 'element' => 'arrows', 'value' => 'true' ),
				'std' => 'default',
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_ofswitch',
				'heading' => esc_html__( 'Dot Navigation', 'total-theme-core' ),
				'param_name' => 'dots',
				'std' => 'false',
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_ofswitch',
				'heading' => esc_html__( 'Auto Play', 'total-theme-core' ),
				'param_name' => 'auto_play',
				'std' => 'false',
				'admin_label' => true,
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_text',
				'heading' => esc_html__( 'Autoplay interval timeout.', 'total-theme-core' ),
				'param_name' => 'timeout_duration',
				'placeholder' => $carousel_defaults['autoplayTimeout'] ?? '5000',
				'description' => esc_html__( 'Time in milliseconds between each auto slide. Default is 5000.', 'total-theme-core' ),
				'dependency' => array( 'element' => 'auto_play', 'value' => 'true' ),
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_ofswitch',
				'heading' => esc_html__( 'Infinite Loop', 'total-theme-core' ),
				'param_name' => 'infinite_loop',
				'std' => 'true',
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_ofswitch',
				'heading' => esc_html__( 'Center Item', 'total-theme-core' ),
				'param_name' => 'center',
				'std' => 'false',
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_text',
				'heading' => esc_html__( 'Animation Speed', 'total-theme-core' ),
				'param_name' => 'animation_speed',
				'placeholder' => $carousel_defaults['smartSpeed'] ?? '250',
				'description' => esc_html__( 'Default is 250 milliseconds. Enter 0.0 to disable.', 'total-theme-core' ),
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
					'label_block' => true,
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_ofswitch',
				'std' => 'false',
				'heading' => esc_html__( 'Auto Width', 'total-theme-core' ),
				'param_name' => 'auto_width',
				'description' => esc_html__( 'If enabled the carousel will display items based on their width showing as many as possible.', 'total-theme-core' ),
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_text',
				'heading' => esc_html__( 'Items To Display', 'total-theme-core' ),
				'param_name' => 'items',
				'placeholder' => $carousel_defaults['items'] ?? '4',
				'dependency' => array( 'element' => 'auto_width', 'value' => 'false' ),
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
					'label_block' => true,
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_ofswitch',
				'std' => 'false',
				'heading' => esc_html__( 'Auto Height?', 'total-theme-core' ),
				'param_name' => 'auto_height',
				'dependency' => array( 'element' => 'items', 'value' => '1' ),
				'description' => esc_html__( 'Allows the carousel to change height based on the active item. This setting is used only when you are displaying 1 item per slide.', 'total-theme-core' ),
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_text',
				'heading' => esc_html__( 'Items To Scrollby', 'total-theme-core' ),
				'param_name' => 'items_scroll',
				'placeholder' => $carousel_defaults['slideBy'] ?? '1',
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
					'label_block' => true,
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_text',
				'heading' => esc_html__( 'Tablet: Items To Display', 'total-theme-core' ),
				'param_name' => 'tablet_items',
				'placeholder' => $carousel_defaults['responsive']['768']['items'] ?? '3',
				'dependency' => array( 'element' => 'auto_width', 'value' => 'false' ),
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
					'label_block' => true,
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_text',
				'heading' => esc_html__( 'Mobile Landscape: Items To Display', 'total-theme-core' ),
				'param_name' => 'mobile_landscape_items',
				'placeholder' => $carousel_defaults['responsive']['480']['items'] ?? '2',
				'dependency' => array( 'element' => 'auto_width', 'value' => 'false' ),
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
					'label_block' => true,
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_text',
				'heading' => esc_html__( 'Mobile Portrait: Items To Display', 'total-theme-core' ),
				'param_name' => 'mobile_portrait_items',
				'placeholder' => $carousel_defaults['responsive']['0']['items'] ?? '1',
				'dependency' => array( 'element' => 'auto_width', 'value' => 'false' ),
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
					'label_block' => true,
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
			array(
				'type' => 'vcex_text',
				'heading' => esc_html__( 'Margin Between Items', 'total-theme-core' ),
				'description' => vcex_shortcode_param_description( 'px' ),
				'param_name' => 'items_margin',
				'placeholder' => $carousel_defaults['margin'] ?? '15',
				'elementor' => [
					'group' => esc_html__( 'Carousel Settings', 'total-theme-core' ),
					'label_block' => true,
				],
				'editors' => [ 'wpbakery', 'elementor' ],
			),
		);

		if ( $dependency ) {
			foreach ( $params as $key => $value ) {
				if ( empty( $params[$key]['dependency'] ) ) {
					$params[$key]['dependency'] = $dependency;
				}
			}
		}

		if ( $group ) {
			foreach ( $params as $key => $value ) {
				$params[$key]['group'] = $group;
			}
		}

		return $params;
	}

	/**
	 * Return carousel default settings.
	 */
	public static function get_default_settings() {
		$settings = [
			'nav'                  => 'true',
			'dots'                 => 'false',
			'autoplay'             => 'false',
			'animateIn'            => 'false',
			'animateOut'           => 'false',
			'lazyLoad'             => 'false',
			'loop'                 => 'true',
			'autoplayHoverPause'   => 'true',
			'center'               => 'false',
			'smartSpeed'           => '250',
			'slideBy'              => '1',
			'autoplayTimeout'      => '5000',
			'margin'               => '15',
			'items'                => '4',
			'autoHeight'           => 'false',
			'autoWidth'            => 'false',
			'rtl'                  => is_rtl(),
			'navClass'             => [
				'owl-nav__btn owl-prev',
				'owl-nav__btn owl-next',
			],
			'responsive'           => [
				'0' => [
					'items' => '1',
				],
				'480' => [
					'items' => '2',
				],
				'768' => [
					'items' => '3',
				],
				'960' => [
					'items' => '4',
				],
			],
		//	'prevIcon'             => self::get_prev_icon(), //@todo update to standardize icons and use SVGs.
		//	'nextIcon'             => self::get_next_icon(), //@todo update to standardize icons and use SVGs.
			'i18n'                 => [
				'next' => esc_html__( 'Next Slide', 'total' ),
				'prev' => esc_html__( 'Previous Slide', 'total' ),
			],
		];

		$settings = (array) apply_filters( 'vcex_carousel_default_settings', $settings );

		return self::parse_booleans( $settings );
	}

	/**
	 * Return carousel settings.
	 */
	public static function get_settings( $atts, $shortcode ) {
		$defaults = self::get_default_settings();

		$settings = [
			'nav'                  => self::parse_setting_bool( 'arrows', $atts, 'true' ),
			'dots'                 => self::parse_setting_bool( 'dots', $atts, 'false' ),
			'autoplay'             => self::parse_setting_bool( 'auto_play', $atts, 'false' ),
			'loop'                 => self::parse_setting_bool( 'infinite_loop', $atts, 'true' ),
			'center'               => self::parse_setting_bool( 'center', $atts, 'false' ),
			'smartSpeed'           => ! empty( $atts['animation_speed'] ) ? floatval( $atts['animation_speed'] ) : '250',
			'items'                => ! empty( $atts['items'] ) ? absint( $atts['items'] ) : '4',
			'slideBy'              => ! empty( $atts['items_scroll'] ) ? absint( $atts['items_scroll'] ) : '1',
			'autoplayTimeout'      => ! empty( $atts['timeout_duration'] ) ? absint( $atts['timeout_duration'] ) : '5000', // cant be 0
			'margin'               => ! empty( $atts['items_margin'] ) ? absint( $atts['items_margin'] ) : '15',

			// We need to redefine these so the Vcex\Carousel\Inline_CSS class works,
			'responsive'           => $defaults['responsive'] ?? [],
		];

		if ( ! empty( $atts['mobile_portrait_items'] ) ) {
			$settings['responsive']['0']['items'] = absint( $atts['mobile_portrait_items'] );
		}

		if ( ! empty( $atts['mobile_landscape_items'] ) ) {
			$settings['responsive']['480']['items'] = absint( $atts['mobile_landscape_items'] );
		}

		if ( ! empty( $atts['tablet_items'] ) ) {
			$settings['responsive']['768']['items'] = absint( $atts['tablet_items'] );
		}

		if ( ! empty( $atts['items'] ) ) {
			$settings['responsive']['960']['items'] = absint( $atts['items'] );
		}

		if ( ! empty( $atts['style'] ) && $atts['style'] == 'no-margins' ) {
			$settings['margin'] = 0;
		}

		if ( ! empty( $atts['auto_width'] ) && 1 !== $settings['items'] ) {
			$settings['autoWidth'] = self::parse_setting_bool( 'auto_width', $atts );
		}

		if ( ! empty( $atts['auto_height'] ) ) {
			$settings['autoHeight'] = self::parse_setting_bool( 'auto_height', $atts );
		}

		$settings = (array) apply_filters( 'vcex_get_carousel_settings', $settings, $atts, $shortcode );

		/**
		 * Filters the carousel settings.
		 *
		 * @param array $settings
		 * @param array $shortcode_attributes
		 * @param string $shortcode
		 */
		$settings = (array) apply_filters( 'vcex_carousel_settings', $settings, $atts, $shortcode );

		return self::parse_settings( $settings );
	}

	/**
	 * Parses a boolean setting to make sure it's on/off.
	 */
	protected static function parse_setting_bool( $key = '', $atts = [], $default = '' ) {
		if ( isset( $atts['is_elementor_widget'] )
			&& true === $atts['is_elementor_widget']
			&& isset( $atts[$key] )
			&& '' === $atts[$key]
		) {
			return 'false';
		}
		if ( empty( $atts[$key] ) || '' === $atts[$key] ) {
			return $default;
		}
		return $atts[$key];
	}

	/**
	 * Parses booleans to ensure they are strings.
	 */
	protected static function parse_booleans( $settings = [] ) {
		if ( is_array( $settings ) ) {
			foreach ( $settings as $key => $val ) {
				if ( is_bool( $val ) ) {
					$settings[$key] = $val ? 'true' : 'false'; // settings must be strings and not actual booleans.
				}
			}
		}
		return $settings;
	}

	/**
	 * Parses settings array.
	 */
	protected static function parse_settings( $settings = [] ) {
		// Make sure booleans are strings.
		$settings = self::parse_booleans( $settings );

		// Checks deprecated params.
		if ( ! empty( $settings['itemsMobilePortrait'] ) ) {
			$settings['responsive']['0']['items'] = $settings['itemsMobilePortrait'];
			unset( $settings['itemsMobilePortrait'] );
		}

		if ( ! empty( $settings['itemsMobileLandscape'] ) ) {
			$settings['responsive']['480']['items'] = $settings['itemsMobileLandscape'];
			unset( $settings['itemsMobileLandscape'] );
		}

		if ( ! empty( $settings['itemsTablet'] ) ) {
			$settings['responsive']['768']['items'] = $settings['itemsTablet'];
			unset( $settings['itemsTablet'] );
		}

		// Return parsed settings.
		return $settings;
	}

	/**
	 * Return carousel settings in json format.
	 */
	public static function get_settings_json( $atts, $shortcode ) {
		return self::to_json( self::get_settings( $atts, $shortcode ) );
	}

	/**
	 * Loops through array to remove duplicates and leaves only different values.
	 */
	protected static function remove_array_dups( $new_settings = [], $defaults = [] ) {
		foreach ( $new_settings as $setting_k => $setting_v ) {
			if ( ! isset( $defaults[$setting_k] ) ) {
				continue;
			}
			if ( is_array( $setting_v ) ) {
				$new_sub_settings = self::remove_array_dups( $setting_v, $defaults[$setting_k] );
				if ( $new_sub_settings ) {
					$new_settings[$setting_k] = $new_sub_settings;
				} else {
					unset( $new_settings[$setting_k] );
				}
			} else {
				if ( $defaults[$setting_k] === $setting_v ) {
					unset( $new_settings[$setting_k] );
				}
			}
		}
		return $new_settings;
	}

	/**
	 * Parses settings array and converts to json.
	 */
	public static function to_json( $settings = [] ) {
		if ( ! $settings ) {
			return;
		}
		$defaults = self::get_default_settings();
		$settings = self::remove_array_dups( $settings, $defaults );
		if ( $settings ) {
			return esc_attr( wp_json_encode( $settings ) );
		}
	}

	/**
	 * Returns prev icon html.
	 */
	protected static function get_prev_icon() {
		if ( is_callable( 'TotalTheme\\Theme_Icons::get_icon' ) ) {
			return Theme_Icons::get_icon( 'material/arrow-back-ios' );
		}
	}

	/**
	 * Returns next icon html.
	 */
	protected static function get_next_icon() {
		if ( is_callable( 'TotalTheme\\Theme_Icons::get_icon' ) ) {
			return Theme_Icons::get_icon( 'chevron-right' );
		}
	}
}