/* jshint esversion: 6 */
( function() {

	let isDoingAjax = false;
	const isLoggedIn = document.querySelector( 'body' ).classList.contains( 'logged-in' );

	// -------------------------- HELPERS -------------------------- //

	/**
	 * Debounce function.
	 */
	const debounce = (callback, wait = 10) => {
		let timeoutId = null;
		return (...args) => {
			window.clearTimeout(timeoutId);
			timeoutId = window.setTimeout(() => {
			  callback.apply(null, args);
			}, wait);
		};
	};

	/**
	 *  Checks if sessionStorage is supported.
	 */
	const useStorage = () => {
		return parseInt( vcex_ajax_params.session_storage );
	};

	/**
	 * Returns a string version of a filter.
	 */
	const shortcodeAttsToString = ( shortcodeAtts ) => {
		return JSON.stringify( shortcodeAtts );
	};

	/**
	 * Maybe adds filter to Session Storage.
	 */
	const storeRequest = ( key, response ) => {
		if ( null === localStorage.getItem( key ) ) {
			sessionStorage.setItem( key, JSON.stringify( response ) );
		}
	};

	/**
	 * Gets filter from Session storage.
	 */
	const getStoredRequest = ( key ) => {
		const storage = sessionStorage.getItem( key );
		if ( storage ) {
			return JSON.parse( storage );
		}
	};

	/**
	 * Handles the AJAX request.
	 */
	const ajaxRequest = ( shortcodeClass, shortcodeAtts, extraData, store = true ) => {
		if ( ! useStorage() ) {
			store = false;
		}

		const shortcodeAttsString = store ? shortcodeAttsToString( shortcodeAtts ) : '';
		const storage = store ? getStoredRequest( shortcodeAttsString ) : '';

		if ( storage ) {
			return new Promise((resolve, reject) => {
				setTimeout(() => {
					resolve(storage);
				}, 50 );
			});
		} else {
			const data = new FormData();
			data.append( 'action', 'vcex_ajax_action' );
			data.append( 'nonce', vcex_ajax_params.nonce );
			data.append( 'shortcodeClass', shortcodeClass );
			data.append( 'shortcodeAtts', JSON.stringify( shortcodeAtts ) );
			if ( extraData ) {
				data.append( 'data', JSON.stringify( extraData ) );
			}
			return fetch( vcex_ajax_params.ajax_url, {
				method: 'POST',
				credentials: 'same-origin',
				body: data,
			} )
			.then( (response) => response.json() )
			.then( (json) => {
				if ( store ) {
					storeRequest( shortcodeAttsString, json.data );
				}
				return json.data;
			})
			.catch( ( error ) => {
				console.log( '[TotalTheme Core Load More]' );
				console.error( error );
			} );
		}
	};

	/**
	 * Checks if jQuery is loaded.
	 */
	const getGridFilterData = ( grid ) => {
		const gridId = grid.id;
		if ( ! gridId ) {
			return;
		}
		const selection = grid.dataset.vcexFilterSelection;
		if ( ! selection ) {
			return;
		}
		const data = {
			'selection': JSON.parse( selection ),
		};

		const filter = document.querySelector( `[data-vcex-ajax-filter-target="${gridId}"]` );

		if ( ! filter ) {
			return data;
		}

		if ( filter.dataset.vcexAjaxFilterMultiple ) {
			data.multiple = 1;
			if ( filter.dataset.vcexAjaxFilterRelation ) {
				data.relation = filter.dataset.vcexAjaxFilterRelation;
			}
			if ( filter.dataset.vcexAjaxFilterMetaRelation ) {
				data.meta_relation = filter.dataset.vcexAjaxFilterMetaRelation;
			}
		}

		if ( filter.dataset.vcexAjaxFilterIgnoreTaxQuery ) {
			data.ignore_tax_query = filter.dataset.vcexAjaxFilterIgnoreTaxQuery;
		}

		return data;
	};

	/**
	 * Checks if jQuery is loaded.
	 */
	const jQueryLoaded = () => {
		return ( 'function' === typeof jQuery );
	};

	/**
	 * Checks if the grid is using isotope.
	 */
	const isotopeCheck = ( grid ) => {
		if ( ! grid || 'function' !== typeof Isotope ) {
			return;
		}
		if ( grid.classList.contains( 'vcex-isotope-grid' ) || grid.classList.contains( 'vcex-navbar-filter-grid' ) || grid.classList.contains( 'wpex-masonry-grid' ) ) {
			return true;
		}
	};

	/**
	 * Refreshes waypoints to prevent issues with CSS animations.
	 */
	const refreshWaypoints = () => {
		if ( 'function' === typeof VcWaypoint ) {
			VcWaypoint.refreshAll();
		}
		if ( 'function' === typeof Waypoint ) {
			Waypoint.refreshAll();
		}
	};

	/**
	 * Re-runs JS functions whenever content is added via ajax.
	 */
	const reloadJS = ( context ) => {
		if ( 'object' === typeof wpex && 'function' === typeof wpex.equalHeights ) {
			wpex.equalHeights( context );
		}

		if ( 'object' === typeof wpex ) {
			if ( 'function' === typeof wpex.hoverStyles ) {
				wpex.hoverStyles();
			}
		}

		if ( 'function' === typeof vc_waypoints ) {
			vc_waypoints();
		} else {
			context.querySelectorAll( '.wpb_animate_when_almost_visible' ).forEach( ( element ) => {
				element.classList.add( 'wpb_start_animation animated' );
			} );
		}

		if ( 'function' === typeof wpexIsotope && context.querySelector( '.wpex-masonry-grid' ) ) {
			wpexIsotope( context );
		}

		if ( 'function' === typeof vcexIsotopeGrids && context.querySelector( '.vcex-isotope-grid' ) ) {
			vcexIsotopeGrids( context );
		}

		if ( jQueryLoaded() ) {
			if ( 'function' === typeof window.wpexSliderPro ) {
				window.wpexSliderPro( jQuery( context ) );
			}
			if ( context.classList.contains( 'justified-gallery' ) && 'undefined' !== typeof jQuery.fn.justifiedGallery ) {
				jQuery.justifiedGallery( 'norewind' );
			}
			if ( 'undefined' !== typeof jQuery.fn.mediaelementplayer ) {
				jQuery( context ).find( 'audio, video' ).mediaelementplayer();
			}
			if ( 'function' === typeof window.vcexCarousels ) {
				vcexCarousels( context );
			}
		}

	};

	/**
	 * Replaces an element with ajaxed response.
	 */
	const replaceShortcode = ( element, newHtml ) => {
		return new Promise( ( resolve, reject ) => {
			const newHtmlEl = newHtml.querySelector( '[data-vcex-class]' );

			if ( ! newHtmlEl ) {
				return reject( new Error( '[TotalTheme Core data-vcex-class not found]' ) );
			}

			const replaceHTML = () => {
				element.innerHTML = newHtmlEl.innerHTML;

				// Re-run js functions.
				reloadJS( element );

				if ( 'function' === typeof vcexNavbarFilterLinks ) {
					vcexNavbarFilterLinks();
				}

				refreshWaypoints();
				resolve( element );
			};

			if ( 'function' === typeof imagesLoaded ) {
				imagesLoaded( newHtml, function( instance ) {
					return replaceHTML();
				} );
			} else {
				return replaceHTML();
			}

		} );
	};

	/**
	 * Inserts new items into the shortcode.
	 */
	const insertPosts = ( grid, posts ) => {

		return new Promise( ( resolve, reject ) => {

			const insert = () => {

				// Adds elements to the grid.
				posts.forEach( ( element ) => {
					if ( element.classList.contains( 'sticky' ) ) {
						element.classList.add( 'vcex-duplicate' ); // @todo deprecate?
					}
					grid.appendChild( element );
				} );

				// Re-run js functions.
				reloadJS( grid );

				// Fixed isotope Layouts.
				if ( isotopeCheck( grid ) ) {
					var isotope = Isotope.data( grid );
					if ( isotope ) {
						isotope.appended( posts );
						isotope.layout();
					}
				}

				refreshWaypoints();
				resolve( posts );
			};

			if ( 'function' === typeof imagesLoaded ) {
				imagesLoaded( posts, function( instance ) {
					return insert();
				} );
			} else {
				return insert();
			}

		} );
	};

	// -------------------------- AJAX FILTER -------------------------- //
	if ( 'function' !== typeof window.vcexAjaxFilter ) {
		window.vcexAjaxFilter = function( context ) {
			if ( ! context || ! context.childNodes ) {
				context = document;
			}

			const beforeAjaxRequest = ( grid ) => {
				isDoingAjax = true;
				let loadingClassname = grid.classList.contains( 'wpex-post-cards' ) ? 'wpex-post-cards--loading' : 'vcex-ajax-loading';
				grid.classList.add( loadingClassname );
				const ajaxLoader = grid.querySelector( '.vcex-ajax-loader' );
				if ( ajaxLoader ) {
					ajaxLoader.style.display = 'flex';
				}
			};

			const onAjaxSuccess = ( response, grid ) => {
				const parser  = new DOMParser();
				const newHtml = parser.parseFromString( response, 'text/html' );
				if ( newHtml ) {
					replaceShortcode( grid, newHtml ).then( result => {
						isDoingAjax = false;
						const newWrapper = newHtml.querySelector( '[data-vcex-atts]' );
						const loadingClassname = grid.classList.contains( 'wpex-post-cards' ) ? 'wpex-post-cards--loading' : 'vcex-ajax-loading';
						// @todo remove ajax_filter and ajax_action and re-add the ajax_filter based on vcex-filter-selection in the loadmore/pagination functions.
						grid.setAttribute( 'data-vcex-atts', newWrapper.dataset.vcexAtts );
						grid.setAttribute( 'data-vcex-current-page', '1' );
						grid.setAttribute( 'data-vcex-max-pages', newWrapper.dataset.vcexMaxPages );
						grid.classList.remove( loadingClassname );
						const ajaxLoader = grid.querySelector( '.vcex-ajax-loader' );
						if ( ajaxLoader ) {
							ajaxLoader.style.display = 'none';
						}
					} );
				}
			};

			const getFilterItemFilter = ( item ) => {
				return item.closest( '[data-vcex-ajax-filter]' );
			};

			const getFilterItemType = ( item ) => {
				return item.dataset.vcexType;
			};

			const getFilterItemKey = ( item ) => {
				return item.dataset.vcexKey;
			};

			const getFilterItemValue = ( item ) => {
				return item.dataset.vcexValue;
			};

			const getFilterItemOperator = ( item ) => {
				return item.dataset.vcexOperator || item.dataset.vcexMetaCompare || 'IN';
			};

			const getFilterItemSelectionString = ( item ) => {
				const itemType = getFilterItemType( item );

				if ( isSelectedTypeAString( itemType ) ) {
					return getFilterItemValue( item );
				}

				const stringArray = [
					`val:${getFilterItemValue( item )}`
				];

				const key = getFilterItemKey( item );
				if ( key ) {
					stringArray.push( `key:${key}` );
				}

				const operator = getFilterItemOperator( item );
				if ( operator ) {
					stringArray.push( `op:${operator}` );
				}

				const metaType = item.dataset.vcexMetaType;
				if ( metaType ) {
					stringArray.push( `type:${metaType}` );
				}

				return stringArray.join( '|' );
			};

			const isSelectedTypeAString = ( type ) => {
				const stringTypes = [ 'orderby', 'order', 'search' ];
				return stringTypes.includes( type );
			};

			const getFilterTargetId = ( filter ) => {
				return filter.dataset.vcexAjaxFilterTarget;
			};

			const getFilterTarget = ( filter ) => {
				const targetId = getFilterTargetId( filter );
				if ( targetId ) {
					const target = targetId.replace( '#', '' );
					return document.querySelector( `#${target}`);
				}
			};

			const getFilterSelection = ( filter ) => {
				const target = getFilterTarget( filter );
				const data = target.dataset.vcexFilterSelection;
				return ( data && undefined !== data ) ? JSON.parse( data ) : {};
			};

			const isFilterSelectionEmpty = ( filter ) => {
				const selection = getFilterSelection( filter );
				if ( ! selection ) {
					return true;
				}
				const selectionString = JSON.stringify( selection );
				if ( '{}' === selectionString ) {
					return true;
				}
				return false;
			};

			const createFilterSelection = ( filter ) => {
				getFilterTarget( filter ).setAttribute( 'data-vcex-filter-selection', '{}' );
			};

			const updateFilterSelection = ( filter, newSelection ) => {
				getFilterTarget( filter ).setAttribute( 'data-vcex-filter-selection', JSON.stringify( newSelection ) );
			};

			const filterSelectionAdd = ( item ) => {
				const filter = getFilterItemFilter( item );
				const target = getFilterTarget( filter );

				if ( ! target.dataset.vcexFilterSelection ) {
					createFilterSelection( filter );
				}

				let value = getFilterItemValue( item );

				if ( ! value ) {
					return filterSelectionRemove( item );
				}

				const selection = getFilterSelection( filter );
				const type = getFilterItemType( item );
				const stringType = isSelectedTypeAString( type );
				const operator = getFilterItemOperator( item );

				if ( 'meta' === type ) {
					value = getFilterItemSelectionString( item );
				}

				if ( type in selection ) {
					if ( stringType ) {
						selection[type] = value;
					} else if ( 'meta' == type ) {
						let metaVal = selection[type];
						if ( metaVal.length ) {
							const metaCheck = getMetaStringCheck( value );
							metaVal.forEach( ( metaItem, metaItemIndex ) => {
								let metaItemCheck = getMetaStringCheck( metaItem );
								if ( metaItemCheck === metaCheck ) {
									metaVal[metaItemIndex] = value;
								} else {
									metaVal.push( value );
								}
							} );
						}
						selection[type] = metaVal;
					} else if ( operator ) {
						let typeObj = selection[type];
						if ( operator in typeObj ) {
							let operatorVal = typeObj[operator];
							operatorVal.push( value );
							selection[type][operator] = operatorVal;
						} else {
							if ( 'object' === typeof typeObj ) {
								selection[type][operator] = new Array( value );
							} else {
								selection[type] =  {
									[operator]: new Array( value )
								};
							}
						}
					} else {
						let typeVal = selection[type];
						typeVal.push( value );
						selection[type] = typeVal;
					}
				} else {
					if ( stringType ) {
						selection[type] = value;
					} else if ( operator && 'meta' !== type ) {
						selection[type] = {
							[operator]: new Array( value )
						};
					} else {
						selection[type] = new Array( value );
					}
				}
				target.dataset.vcexFilterSelection = JSON.stringify( selection );
			};

			const getMetaStringCheck = ( string ) => {
				const array = string.split( '|' );
				array.shift(); // remove val
				return array.join( '|' );
			};

			const filterSelectionRemove = ( item, forced ) => {
				let value = getFilterItemValue( item );

				if ( ! value && true !== forced ) {
					return;
				}

				const filter = getFilterItemFilter( item );
				const target = getFilterTarget( filter );
				const selection = getFilterSelection( filter );
				const type = getFilterItemType( item );
				const stringType = isSelectedTypeAString( type );
				const operator = getFilterItemOperator( item );

				if ( 'meta' === type ) {
					value = getFilterItemSelectionString( item );
				}

				if ( type in selection ) {
					if ( stringType ) {
						delete selection[type];
					} else {
						if ( 'meta' == type ) {
							let metaVal = selection[type];
							if ( Array.isArray( metaVal ) ) {
								const metaCheck = getMetaStringCheck( value );
								metaVal.forEach( ( metaItem, metaItemIndex ) => {
									let metaItemCheck = getMetaStringCheck( metaItem );
									if ( metaItemCheck === metaCheck ) {
										metaVal.splice( metaItemIndex, 1);
									}
								} );
							}
							if ( metaVal.length ) {
								selection[type] = metaVal;
							} else {
								delete selection[type];
							}
						} else if ( operator ) {
							if ( operator in selection[type]  ) {
								let operatorVal = selection[type][operator];
								let valIndex = operatorVal.indexOf( value );
								if ( -1 !== valIndex ) {
									operatorVal.splice( valIndex, 1 );
								}
								if ( operatorVal.length ) {
									selection[type][operator] = operatorVal;
								} else {
									delete selection[type];
								}
							}

						} else {
							let typeVal = selection[type];
							let valIndex = typeVal.indexOf( value );
							if ( -1 !== valIndex ) {
								typeVal.splice( valIndex, 1 );
							}
							if ( typeVal.length ) {
								selection[type] = typeVal;
							} else {
								delete selection[type];
							}
						}
					}
				}

				updateFilterSelection( filter, selection );

				if ( isFilterSelectionEmpty( filter ) ) {
					const targetId = getFilterTargetId( filter );
					document.querySelectorAll( `[data-vcex-ajax-filter-target="${targetId}"] [data-vcex-type="all"]`).forEach( ( item ) => {
						activateItem( item );
					} );
				}
			};

			const disableItems = ( filter ) => {
				const targetId = getFilterTargetId( filter );
				document.querySelectorAll( `[data-vcex-ajax-filter-target="${targetId}"]`).forEach( ( el ) => {
					el.style.pointerEvents = 'none';
				} );
			};

			const reEnableItems = ( filter ) => {
				const targetId = getFilterTargetId( filter );
				document.querySelectorAll( `[data-vcex-ajax-filter-target="${targetId}"]`).forEach( ( el ) => {
					el.style.pointerEvents = '';
				} );
			};

			const activateItem = ( item ) => {
				item.classList.add( 'active' );
				item.setAttribute( 'data-vcex-selected', '1' );
				const itemIsAll = isAllItem( item );
				if ( ! itemIsAll ) {
					filterSelectionAdd( item );
				}
				if ( itemIsAll || ! filterSupportsMultiple( getFilterItemFilter( item ) ) ) {
					item.classList.add( 'wpex-pointer-events-none' );
				}
			};

			const deActivateItem = ( item ) => {
				if ( ! isFilterItemActive( item ) ) {
					item.classList.remove( 'active' ); // extra check for active items on load.
					item.removeAttribute( 'data-vcex-selected' );
					return;
				}
				const tag = item.tagName.toLowerCase();
				item.classList.remove( 'active', 'wpex-pointer-events-none' );
				item.removeAttribute( 'data-vcex-selected' );
				if ( ! isAllItem( item ) || 'select' === tag ) {
					filterSelectionRemove( item ); // must run before removing data-vcex-value or it won't work.
				}
				switch ( tag ) {
					case 'select':
						item.removeAttribute( 'data-vcex-value' );
						item.selectedIndex = 0;
						break;
					case 'input':
						const inputType = item.getAttribute( 'type' );
						switch ( inputType ) {
							case 'checkbox':
								item.checked = false;
								break;
							case 'range':
								item.removeAttribute( 'data-vcex-value' );
								break;
							default:
								item.value = '';
								item.removeAttribute( 'data-vcex-value' );
								break;
						}
						break;
				}
			};

			const deActivateAll = ( filter ) => {
				const targetId = getFilterTargetId( filter );
				if ( ! targetId ) {
					return;
				}
				document.querySelectorAll( `[data-vcex-ajax-filter-target="${targetId}"] [data-vcex-type]`).forEach( ( item ) => {
					if ( 'all' === item.dataset.vcexType ) {
						activateItem( item );
					} else {
						deActivateItem( item );
					}
				} );

				// Loop through non vcex inputs that are likely hooked into hidden fields to clear them.
				document.querySelectorAll( `[data-vcex-ajax-filter-target="${targetId}"] input:not([data-vcex-type])`).forEach( ( item ) => {
					const type = item.getAttribute( 'type' );
					switch ( type ) {
						case 'checkbox':
							item.checked = false;
							break;
						case 'select':
							item.selectedIndex = 0;
							break;
						default:
							item.value = '';
							break;
					}
				} );
				createFilterSelection( filter );
			};

			const isAllItem = ( filterItem ) => {
				const value = getFilterItemValue( filterItem );
				const type = getFilterItemType( filterItem );
				return ( 'reset' === type || 'all' === type || 'all' === value ) ? true : false;
			};

			const isFilterItemActive = ( filterItem ) => {
				if ( isAllItem( filterItem ) ) {
					if ( filterItem.classList.contains( 'active' ) || '1' === filterItem.dataset.vcexSelected || isFilterSelectionEmpty( getFilterItemFilter( filterItem ) ) ) {
						return true;
					}
					return false;
				}
				return filterItem.dataset.vcexSelected ? true : false;
				/*
				const filter = getFilterItemFilter( filterItem );
				let selection = getFilterSelection( filter );
				if ( isAllItem( filterItem ) ) {
					return isFilterSelectionEmpty( filter ) ? true : false;
				}
				if ( ! selection ) {
					return false;
				}
				const type = getFilterItemType( filterItem );
				const value = getFilterItemValue( filterItem );
				const operator = getFilterItemOperator( filterItem );
				if ( type in selection ) {
					let typeVal = selection[type];
					if ( operator && 'object' === typeof typeVal ) {
						if ( operator in typeVal ) {
							return ( -1 !== typeVal[operator].indexOf( value ) ) ? true : false;
						}
					} else {
						return ( -1 !== typeVal.indexOf( value ) ) ? true : false;
					}
				}
				*/
			};

			const filterSupportsMultiple = ( filter ) => {
				return parseInt( filter.dataset.vcexAjaxFilterMultiple );
			};

			const getFilterCounters = ( filter ) => {
				const data = {};
				filter.querySelectorAll( '[data-vcex-type] .vcex-ajax-filter-count' ).forEach( ( count ) => {
					const filterItem = count.closest( '[data-vcex-type]' );
					const itemType = getFilterItemType( filterItem );
					if ( ! itemType ) {
						return; // this could be an empty type like an ALL link.
					}
					const value = getFilterItemValue( filterItem );
					const objKey = itemType + '|' + value;
					const number = count.innerText;
					data[objKey] = number.replace(/\D/g,'');
				} );
				return data;
			};

			const getFilterItemsData = ( filter ) => {
				const data = {};
				const targetId = getFilterTargetId( filter );
				document.querySelectorAll( `[data-vcex-ajax-filter-target="${targetId}"] .vcex-ajax-filter-count` ).forEach( ( count ) => {
					const filterItem = count.closest( '[data-vcex-type]' );
					const type = getFilterItemType( filterItem );
					if ( ! type || ! filterItem || 'all' === type ) {
						return;
					}
					const value = getFilterItemValue( filterItem );
					if ( type in data ) {
						data[type].push( value );
					} else {
						data[type] = [ value ];
					}
				} );
				return data;
			};

			const updateCounts = ( filter, newCounts ) => {
				if ( ! newCounts || undefined === newCounts ) {
					return;
				}
				if ( 'object' !== typeof newCounts ) {
					newCounts = JSON.parse( newCounts );
				}
				const countEls = filter.querySelectorAll( '.vcex-ajax-filter-count' );
				if ( ! countEls ) {
					return;
				}
				if ( 'empty' === newCounts || '0' === newCounts ) {
					countEls.forEach( ( count ) => {
						const filterItem = count.closest( '[data-vcex-type]' );
						if ( ! isAllItem( filterItem ) ) {
							count.textContent = '(0)';
							if ( ! isFilterItemActive( filterItem ) ) {
								filterItem.classList.add( 'wpex-pointer-events-none' );
								filterItem.setAttribute( 'disabled', '' );
							}
						}
					} );
				} else {
					for (let [key, newCount] of Object.entries(newCounts)) {
						if ( 'all' === key ) {
							document.querySelectorAll( '[data-vcex-type="all"] .vcex-ajax-filter-count' ).forEach( ( filterCountEl ) => {
								filterCountEl.textContent = `(${newCount})`;
							} );
						} else {
							const splitKey = key.split( '|' );
							const type = splitKey[0];
							const value = splitKey[1];
							document.querySelectorAll( `[data-vcex-type="${type}"][data-vcex-value="${value}"] .vcex-ajax-filter-count` ).forEach( ( filterCountEl ) => {
								filterCountEl.textContent = `(${newCount})`;
								const filterItem = filterCountEl.closest( '[data-vcex-type]' );
								if ( isFilterItemActive( filterItem ) ) {
									return;
								}
								if ( ! newCount || '0' === newCount ) {
									filterItem.classList.add( 'wpex-pointer-events-none' );
									filterItem.setAttribute( 'disabled', '' );
								} else {
									filterItem.classList.remove( 'wpex-pointer-events-none' );
									filterItem.removeAttribute( 'disabled' );
								}
							} );
						}
					}
				}
			};

			const newFilter = ( selectedItem ) => {
				const selectedType = getFilterItemType( selectedItem );
				const selectedValue = getFilterItemValue( selectedItem );

				if ( ! selectedType && ! selectedValue ) {
					return;
				}

				const filter = selectedItem.closest( '[data-vcex-ajax-filter]' );
				const targetId = getFilterTargetId( filter );
				const targetGrid = getFilterTarget( filter );

				if ( ! targetGrid ) {
					return;
				}

				const ajaxExtraData = {};
				const multiple = filterSupportsMultiple( filter );
				const relation = filter.dataset.vcexAjaxFilterRelation;
				const isSelectedItemActive = isFilterItemActive( selectedItem );
				const isSelectedItemAllBtn = isAllItem( selectedItem );
				const filterData = getFilterSelection( filter );
				const selectedItemTag = selectedItem.tagName.toLowerCase();
				const inputType = selectedItem.getAttribute( 'type' );
				const override = ( 'select' === selectedItemTag || ( 'hidden' === inputType || 'range' === inputType ) ) ? true : false;

				if ( ( ! override && ! multiple && isSelectedItemActive ) || ( isSelectedItemAllBtn && isSelectedItemActive ) ) {
					return;
				}

				const shortcodeClass = targetGrid.dataset.vcexClass;
				const shortcodeAtts  = JSON.parse( targetGrid.dataset.vcexAtts );

				if ( ! shortcodeClass || ! shortcodeAtts ) {
					return;
				}

				if ( isSelectedItemAllBtn && isSelectedItemActive ) {
					return;
				}

				// Start new filter here.
				let filterItems = [];
				let showAll = false;
				shortcodeAtts.ajax_filter = {}; // don't "delete" because it changes the order around and then session Storage won't work as expected.

				if ( isSelectedItemAllBtn ) {
					showAll = true;
					deActivateAll( filter );
					activateItem( selectedItem );
					targetGrid.setAttribute( 'data-vcex-current-page', '1' );
				} else {

					if ( ! override && multiple && isFilterItemActive( selectedItem ) ) {
						deActivateItem( selectedItem );
					} else {
						activateItem( selectedItem );
					}

					filter.querySelectorAll( '[data-vcex-type]' ).forEach( ( item ) => {
						if ( item.isSameNode( selectedItem ) ) {
							return;
						}
						if ( multiple ) {
							if ( isAllItem( item ) && ! isFilterSelectionEmpty( filter ) ) {
								deActivateItem( item );
							}
						} else {
							deActivateItem( item );
						}
					} );
					if ( filterItems ) {
						shortcodeAtts.ajax_filter = getGridFilterData( targetGrid );
					}
				}

				if ( ! filterItems && ! showAll ) {
					return;
				}

				shortcodeAtts.ajax_action = 'filter'; // define ajax_action type.

				beforeAjaxRequest( targetGrid );
				disableItems( filter );

				// Filter data used to update counts.
				if ( filter.dataset.vcexAjaxFilterUpdateCounts && document.querySelector( `[data-vcex-ajax-filter-target="${targetId}"] .vcex-ajax-filter-count` ) ) {
					ajaxExtraData.update_counts = '1';
					ajaxExtraData.filter = getFilterItemsData( filter );
				}

				// Make ajax request.
				ajaxRequest( shortcodeClass, shortcodeAtts, ajaxExtraData ).then( ( response ) => {
					if ( response.counts ) {
						updateCounts( filter, response.counts );
					}
					if ( response.html ) {
						onAjaxSuccess( response.html, targetGrid );
					}
					reEnableItems( filter );
				} );
			};

			const onDocClick = ( event ) => {
				if ( isDoingAjax ) {
					return;
				}

				const filterItem = event.target.closest( 'a[data-vcex-type],button[data-vcex-type]' );

				if ( ! filterItem ) {
					return;
				}

				event.preventDefault();

				newFilter( filterItem );
			};

			const onChange = ( event ) => {
				if ( isDoingAjax ) {
					return;
				}

				const target = event.target;

				if ( ! target.getAttribute( 'data-vcex-type' ) ) {
					return;
				}

				const filterItem = target.closest( 'select, input[type="checkbox"], input[type="hidden"], input[type="range"]' );

				if ( ! filterItem ) {
					return;
				}

				event.preventDefault();

				if ( 'select' === filterItem.tagName.toLowerCase() || 'range' === filterItem.getAttribute( 'type' ) ) {
					const value = filterItem.value;
					const prevValue = getFilterItemValue( filterItem );
					if ( prevValue && 'all' !== prevValue ) {
						if ( value && 'all' !== value ) {
							filterSelectionRemove( filterItem );
						} else {
							deActivateItem( filterItem );
						}
					}
					if ( value && 'all' !== value ) {
						filterItem.setAttribute( 'data-vcex-value', value );
					}
				} else {
					if ( 'hidden' === filterItem.getAttribute( 'type' ) && ! getFilterItemValue( filterItem ) ) {
						filterSelectionRemove( filterItem, true );
					}
				}

				newFilter( filterItem );
			};

			const onKeyUpDebounced = debounce( ( event ) => {
				if ( isDoingAjax || event.key.includes( 'F5', 'TAB', 'SHIFT', 'ESC' ) ) {
					return;
				}
				const filterItem = event.target.closest( 'input[data-vcex-type]' );

				if ( ! filterItem ) {
					return;
				}

				const inputType = filterItem.getAttribute( 'type' );

				if ( ! filterItem || 'checkbox' === inputType || 'range' === inputType ) {
					return;
				}

				event.preventDefault();

				filterSelectionRemove( filterItem );

				const value = filterItem.value;

				if ( value ) {
					filterItem.setAttribute( 'data-vcex-value', value );
				} else {
					filterItem.removeAttribute( 'data-vcex-value' );
				}

				newFilter( filterItem );
			}, 500);

			const onLoad = () => {
				document.querySelectorAll( '[data-vcex-ajax-filter]' ).forEach( ( filter ) => {
					let shortcodeAttsString;
					let targetGrid = getFilterTarget( filter );

					if ( ! targetGrid ) {
						return;
					}

					const shortcodeClass = targetGrid.dataset.vcexClass;
					const shortcodeAtts = JSON.parse( targetGrid.dataset.vcexAtts );
					shortcodeAtts.ajax_filter = {}; // ADD first to prevent storage issues.
					shortcodeAtts.ajax_action = 'filter';

					if ( ! shortcodeAtts || ! shortcodeClass ) {
						return;
					}

					if ( useStorage() ) {
						shortcodeAttsString = shortcodeAttsToString( shortcodeAtts );
					}

					const storeInitialSetup = () => {
						const storage = getStoredRequest( shortcodeAttsString );
						if ( storage && 'undefined' !== storage ) {
							return;
						}
						data = {
							'html': targetGrid.outerHTML
						};
						if ( filter.dataset.vcexAjaxFilterSetCounts ) {
							data.counts = getFilterCounters( filter );
						}
						storeRequest( shortcodeAttsString, data );
					};

					if ( filter.dataset.vcexAjaxFilterSetCounts ) {
						let storage = useStorage() ? getStoredRequest( 'counts__' + shortcodeAttsString ) : '';
						const ajaxExtraData = {
							'return': 'filter_counts',
							'update_counts': '1',
							'filter': getFilterItemsData( filter ),
						};
						if ( filter.dataset.vcexAjaxFilterIgnoreTaxQuery ) {
							ajaxExtraData.ignore_tax_query = filter.dataset.vcexAjaxFilterIgnoreTaxQuery;
						}
						if ( storage ) {
							updateCounts( filter, storage );
							storeInitialSetup();
						} else {
							disableItems( filter );
							ajaxRequest( shortcodeClass, shortcodeAtts, ajaxExtraData, false ).then( ( response ) => {
								updateCounts( filter, response.counts );
								if ( useStorage() ) {
									storeRequest( 'counts__' + shortcodeAttsString, response.counts );
									storeInitialSetup();
								}
								reEnableItems( filter );
							} );
						}
					} else {
						storeInitialSetup();
					}
				} );

				// Check for ?filter= URL string.
				const queryString = window.location.search;
				const urlParams   = new URLSearchParams( queryString );
				let filter        = urlParams.get( 'filter' );
				if ( ! filter ) {
					return;
				}
				const filterItem = document.querySelector( `[data-vcex-ajax-filter] [data-vcex-name="${filter}"]` );
				if ( filterItem ) {
					newFilter( filterItem );
				}
			};

			// Add event listeners.
			onLoad();
			document.addEventListener( 'click', onDocClick );
			document.addEventListener( 'change', onChange );
			window.addEventListener( 'keyup', onKeyUpDebounced );

			document.querySelectorAll( 'input[data-vcex-type][type="hidden"]' ).forEach( ( hiddenInput ) => {
				hiddenInput.addEventListener( 'change', onChange );
			} );

		};
	}

	// -------------------------- NUMBERED PAGINATION -------------------------- //

	if ( 'function' !== typeof window.vcexAjaxPagination ) {
		window.vcexAjaxPagination = function( context ) {
			if ( ! context || ! context.childNodes ) {
				context = document;
			}

			const onDocClick = ( event ) => {
				const link = event.target.closest( '[data-vcex-pagination="numbered_ajax"] a.page-numbers' );
				if ( ! link ) {
					return;
				}

				event.preventDefault();

				if ( isDoingAjax ) {
					return; // prevent click spam.
				}

				const grid = link.closest( '[data-vcex-pagination="numbered_ajax"]' );
				const pagination = grid.querySelector( '.wpex-pagination' );

				if ( ! pagination ) {
					return;
				}

				const shortcodeClass = grid.dataset.vcexClass;
				const postsHolder = grid.querySelector( '.wpex-post-cards-list, .wpex-post-cards-grid, .wpex-post-cards-flex' );
				let shortcodeAtts = JSON.parse( grid.dataset.vcexAtts );
				let page = parseInt( grid.dataset.vcexCurrentPage );
				let changeFocus = false;
				let loadingClassname = grid.classList.contains( 'wpex-post-cards' ) ? 'wpex-post-cards--loading' : 'vcex-ajax-loading';

				const onAjaxSuccess = ( response ) => {
					const parser = new DOMParser();
					const newHtml = parser.parseFromString( response, 'text/html' );

					if ( newHtml ) {
						replaceShortcode( grid, newHtml ).then( result => {
							shortcodeAtts = JSON.parse( newHtml.querySelector( '[data-vcex-atts]' ).dataset.vcexAtts );
							isDoingAjax = false;
							grid.classList.remove( loadingClassname );
							const ajaxLoader = grid.querySelector( '.vcex-ajax-loader' );
							if ( ajaxLoader ) {
								ajaxLoader.style.display = 'none';
							}
							if ( changeFocus ) {
								const firstLink = grid.querySelector( 'a' );
								if ( firstLink ) {
									firstLink.focus( { focusVisible: false } );
								}
							}
						} );
					}
				};

				if ( link.classList.contains( 'next' ) ) {
					page = page + 1;
				} else if ( link.classList.contains( 'prev' ) ) {
					page = page - 1;
				} else {
					page = parseInt( link.innerText.replace(/\D/g,'') );
				}

				shortcodeAtts.paged = page;
				grid.setAttribute( 'data-vcex-current-page', page );
				isDoingAjax = true;
				grid.classList.add( loadingClassname );

				const ajaxLoader = grid.querySelector( '.vcex-ajax-loader' );

				if ( ajaxLoader ) {
					ajaxLoader.style.display = 'flex';
				}

				shortcodeAtts.ajax_action = 'pagination'; // define ajax_action type.

				const filter_data = getGridFilterData( grid );

				if ( filter_data ) {
					shortcodeAtts.ajax_filter = filter_data;
				}

				ajaxRequest( shortcodeClass, shortcodeAtts ).then( response => {
					if ( response.html ) {
						onAjaxSuccess( response.html );
					}
				} );
			};

			const onload = () => {
				// Save the first page in storage.
				if ( useStorage() ) {
					document.querySelectorAll( '[data-vcex-pagination="numbered_ajax"]' ).forEach( ( grid ) => {
						const shortcodeAtts = JSON.parse( grid.dataset.vcexAtts );
						if ( shortcodeAtts ) {
							shortcodeAtts.paged = 1;
							shortcodeAtts.ajax_action = 'pagination';
							storeRequest( shortcodeAttsToString( shortcodeAtts ), { 'html': grid.outerHTML } );
						}
					} );
				}
			};

			/**
			 * Registers events.
			 */
			onload();
			document.addEventListener( 'click', onDocClick );
		};
	}

	// -------------------------- LOAD MORE -------------------------- //

	if ( 'function' !== typeof window.vcexAjaxLoadmore ) {
		window.vcexAjaxLoadmore = function( context ) {
			if ( ! context || ! context.childNodes ) {
				context = document;
			}

			/**
			 * Handles the button click event.
			 */
			const onDocClick = ( event ) => {
				button = event.target.closest( '.vcex-loadmore-button' );

				if ( ! button ) {
					return;
				}

				event.preventDefault();

				if ( isDoingAjax ) {
					return; // prevent button clicking SPAM.
				}

				const buttonWrap = button.closest( '.vcex-loadmore' );

				if ( ! buttonWrap ) {
					return;
				}

				const wrapper = buttonWrap.closest( '[data-vcex-class]' );

				if ( ! wrapper ) {
					return;
				}

				const grid = wrapper.querySelector( '.wpex-post-cards-list, .wpex-post-cards-grid, .wpex-post-cards-flex' );

				if ( ! grid ) {
					return;
				}

				/**
				 * Handles a successful ajax response.
				 */
				const onAjaxSuccess = ( response ) => {
					const parser = new DOMParser();
					const newHtml = parser.parseFromString( response, 'text/html' );
					const newPosts = newHtml.querySelectorAll( '.wpex-post-cards-entry, .wpex-card-list-divider' );

					if ( newPosts ) {
						wrapper.setAttribute( 'data-vcex-current-page', page );

						insertPosts( grid, newPosts ).then( result => {

							if ( jQueryLoaded() ) {
								jQuery( grid ).trigger( 'vcexAjaxLoadmoreFinished', [ jQuery( newPosts ) ] );
							}

							shortcodeAtts = JSON.parse( newHtml.querySelector( '[data-vcex-atts]' ).dataset.vcexAtts );

							// Update elements.
							if ( infiniteScroll ) {
								buttonWrap.classList.add( 'wpex-invisible' );
							}

							buttonWrap.classList.remove( 'vcex-loading' );
							buttonTextEl.innerText = buttonText;

							// Focus on first element after loading more items for accessibility.
							// @todo can this be done when hitting enter only?
							if ( ! infiniteScroll && changeFocus ) {
								const firstLink = result[0].querySelector( 'a' );
								if ( firstLink ) {
									firstLink.focus();
								}
							}

							if ( page === maxPages ) {
								buttonWrap.style.display = 'none';
								button.setAttribute( 'aria-disabled', 'true' );
							}

							isDoingAjax = false;
						} );
					}
				};

				// Define main variables.
				let page = parseInt( wrapper.dataset.vcexCurrentPage ) + 1;
				let shortcodeAtts = JSON.parse( wrapper.dataset.vcexAtts );
				let changeFocus = false;
				const shortcodeClass = wrapper.dataset.vcexClass;
				const maxPages = parseInt( wrapper.dataset.vcexMaxPages );
				const buttonTextEl = button.querySelector( '.vcex-txt' );
				const buttonText = button.dataset.text;
				const loadingText = button.dataset.loadingText;
				const failedText = button.dataset.failedText;
				const infiniteScroll = parseInt( button.dataset.infiniteScroll );

				// Prevents jump when showing loader icon.
				if ( ! infiniteScroll ) {
					buttonWrap.style.minHeight = buttonWrap.offsetHeight + 'px';
				}


				// Update variables.
				isDoingAjax = true;
				shortcodeAtts.paged = page; // update paged value every time we click the load more button.

				// Tweak some elements.
				if ( infiniteScroll ) {
					buttonWrap.classList.remove( 'wpex-invisible' );
				}

				buttonWrap.classList.add( 'vcex-loading' );
				buttonTextEl.innerText = loadingText;

				// Check if we should change focus on success.
				changeFocus = true;

				shortcodeAtts.ajax_action = 'load_more'; // define ajax_action type

				const filter_data = getGridFilterData( wrapper );

				if ( filter_data ) {
					shortcodeAtts.ajax_filter = filter_data;
				}

				// Lets request items.
				ajaxRequest( shortcodeClass, shortcodeAtts ).then( response => {
					if ( response.html ) {
						onAjaxSuccess( response.html );
					} else {
						buttonTextEl.innerText = failedText;
					}
				} );
			};

			/**
			 * Registers events.
			 */
			document.addEventListener( 'click', onDocClick );
		};
	}

	// -------------------------- INFINITE SCROLL -------------------------- //

	if ( 'function' !== typeof window.vcexAjaxInfiniteScroll ) {
		window.vcexAjaxInfiniteScroll = function() {
			let scrollPos = 0;

			if ( ! document.querySelector( '.vcex-loadmore--infinite-scroll .vcex-loadmore-button' ) ) {
				return;
			}

			const winTop = () => {
				let scrollTop = window.pageYOffset || document.documentElement.scrollTop || document.body.scrollTop || 0;
				if ( scrollTop < 0 ) {
					scrollTop = 0; // return 0 if negative to prevent issues with elastic scrolling in Safari.
				}
				return scrollTop;
			};

			const offset = ( element ) => {
				var rect = element.getBoundingClientRect();
				return {
					top: rect.top + winTop(),
					left: rect.left + winTop(),
				};
			};

			const isInViewport = ( element ) => {
				var rect = element.getBoundingClientRect();
				var html = document.documentElement;
				return (
					rect.bottom <= (window.innerHeight || html.clientHeight) &&
					rect.right <= (window.innerWidth || html.clientWidth)
				);
			};

			const scrollMonitor = debounce( (ev) => {
				let windowY = window.scrollY;
				if ( windowY > scrollPos ) {
					// Note store targets above because of ajax loading.
					document.querySelectorAll( '.vcex-loadmore--infinite-scroll .vcex-loadmore-button' ).forEach( ( button ) => {
						if ( 'true' !== button.getAttribute( 'aria-disabled' ) && isInViewport( button ) ) {
							button.click();
						}
					} );
				}
				scrollPos = windowY;
			}, 10);

			window.addEventListener( 'scroll', scrollMonitor );
		};
	}

	// -------------------------- GET THINGS STARTED -------------------------- //
	const onDocReady = () => {
		vcexAjaxFilter();
	};

	const onWindowLoad = () => {
		vcexAjaxInfiniteScroll();
		vcexAjaxPagination();
		vcexAjaxLoadmore();
	};

	if ( document.readyState === 'interactive' || document.readyState === 'complete' ) {
		setTimeout( onDocReady, 0 );
	} else {
		document.addEventListener( 'DOMContentLoaded', onDocReady, false );
	}

	window.addEventListener( 'load', onWindowLoad );

} ) ();