<?php
namespace TotalThemeCore\Meta;

use WP_Query;
use TotalTheme\Theme_Builder;

defined( 'ABSPATH' ) || exit;

/**
 * Theme Settings Metabox.
 *
 * @package TotalThemeCore
 * @version 1.6.0
 */
final class Theme_Settings {

	/*
	 * Array of post types to display metabox on.
	 *
	 * @var array $post_types
	 */
	private $post_types = [];

	/**
	 * Our single Theme_Settings instance.
	 */
	private static $instance;

	/**
	 * Create or retrieve the instance of Theme_Settings.
	 */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new self();
		}
		return static::$instance;
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'admin_init', array( $this, 'admin_init' ) );
	}

	/**
	 * Start things up on init so we can hook into the various filters.
	 */
	public function admin_init() {
		$this->post_types = array(
			'post'         => 'post',
			'page'         => 'page',
			'portfolio'    => 'portfolio',
			'staff'        => 'staff',
			'testimonials' => 'testimonials',
			'product'      => 'product',
		);

		/**
		 * Filters the post types the Theme Settings metabox should be added to.
		 *
		 * @array $post_types
		 */
		$this->post_types = apply_filters( 'wpex_main_metaboxes_post_types', $this->post_types );

		if ( defined( 'TYPES_VERSION' ) ) {
			$this->add_toolset_types(); // toolset plugin support
		}

		if ( $this->post_types ) {
			foreach( $this->post_types as $key => $val ) {
				add_action( 'add_meta_boxes_' . $val, array( $this, 'post_meta' ), 11 );
			}
		}

		add_action( 'save_post', array( $this, 'save_meta_data' ) );

	}

	/**
	 * The function responsible for creating the actual meta box.
	 */
	public function post_meta( $post ) {
		if ( $this->is_footer_builder_page( $post ) ) {
			return;
		}

		$settings = $this->meta_array( $post );

		if ( ! $settings ) {
			return;
		}

		$tab_count = 0;
		foreach ( $settings as $setting_tab => $setting_args ) {
			$add_to_count = false;
			$post_type = $setting_args['post_type'] ?? null;
			if ( $post_type ) {
				if ( is_string( $post_type ) ) {
					if ( $post->post_type === $post_type ) {
						$add_to_count = true;
					}
				} elseif ( is_array( $post_type ) ) {
					if ( in_array( $post->post_type, $post_type ) ) {
						$add_to_count = true;
					}
				}
			} else {
				$add_to_count = true;
			}
			if ( $add_to_count ) {
				$tab_count++;
			}
		}

		if ( ! $tab_count ) {
			return;
		}

		if ( 1 === $tab_count ) {
			$post_type_obj = get_post_type_object( get_post_type( $post ) );
			$label = $post_type_obj->labels->singular_name ?? '';
			$metabox_title = sprintf( esc_html__( '%s Settings', 'total-theme-core' ), $label );
		} else {
			$metabox_title = esc_html__( 'Theme Settings', 'total-theme-core' );
		}

		add_meta_box(
			'wpex-metabox',
			$metabox_title,
			array( $this, 'display_meta_box' ),
			$post->post_type,
			'normal',
			'high'
		);
	}

	/**
	 * Enqueue scripts and styles needed for the metaboxes.
	 */
	public function load_scripts() {

		// Enqueue metabox css.
		wp_enqueue_style(
			'wpex-post-metabox',
			TTC_PLUGIN_DIR_URL . 'assets/css/metabox.css',
			[],
			'1.0'
		);

		// Enqueue media js.
		wp_enqueue_media();

		// Enqueue color picker.
		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'wp-color-picker' );

		// Load alpa color picker if Nextgen is not active because it breaks things.
		if ( apply_filters( 'wpex_metabox_alpha_color_picker', true )
			&& ! class_exists( 'C_NextGEN_Bootstrap' )
		) {
			wp_enqueue_script( 'wp-color-picker-alpha' );
		}

		// Enqueue metabox js.
		wp_enqueue_script(
			'wpex-post-metabox',
			TTC_PLUGIN_DIR_URL . 'assets/js/metabox.min.js',
			array( 'jquery', 'wp-color-picker' ),
			'1.0',
			true
		);

		wp_localize_script( 'wpex-post-metabox', 'wpexMB', array(
			'reset'  => esc_html__(  'Reset Settings', 'total-theme-core' ),
			'cancel' => esc_html__(  'Cancel Reset', 'total-theme-core' ),
		) );

	}

	/**
	 * Renders the content of the meta box.
	 */
	public function display_meta_box( $post ) {
		$post_id      = $post->ID;
		$post_type    = get_post_type();
		$metabox_html = '';

		// Get tabs
		$tabs = $this->meta_array( $post );

		// Make sure tabs aren't empty.
		if ( empty( $tabs ) ) {
			return;
		}

		// What tab should be open by default.
		$open_tab = '';

		// Store tabs that should display on this specific page in an array for use later.
		$active_tabs = [];
		foreach ( $tabs as $tab_key => $tab ) {

			$tab_post_type = $this->get_tab_post_type( $tab );

			if ( ! $tab_post_type || in_array( $post_type, $tab_post_type ) ) {
				$active_tabs[$tab_key] = $tab;
			}

			if ( $tab_post_type && in_array( $post_type, $tab_post_type ) ) {
				$open_tab = $tab_key;
			}

		}

		// No active tabs.
		if ( empty( $active_tabs ) ) {
			return;
		}

		// Add an nonce field so we can check for it later.
		wp_nonce_field( 'wpex_metabox', 'wpex_metabox_nonce' );

		// Enqueue scripts needed for metabox.
		$this->load_scripts();

		// Get tab count.
		$tab_count = count( $active_tabs );

		if ( $tab_count > 1 ) {

			$tabs_output = '';

			$tabs_output .= '<ul class="wp-tab-bar">';

				$count = 0;
				foreach ( $active_tabs as $tab_key => $tab ) {

					$count++;

					$tab_title = $tab['title'] ?? esc_html__( 'Other', 'total-theme-core' );
					$tab_post_type = $this->get_tab_post_type( $tab );
					$is_open_tab   = $has_active_tab = false;

					if ( ! $has_active_tab ) {
						if ( $open_tab ) {
							if ( $open_tab === $tab_key ) {
								$is_open_tab = true;
							}
						} elseif ( 1 === $count ) {
							$is_open_tab = true;
						}
					}

					$tabs_output .= '<li';

						if ( $is_open_tab && ! $has_active_tab ) {
							$tabs_output .= ' class="wp-tab-active"';
							$has_active_tab = true;
						}

					$tabs_output .= '>';

						$tabs_output .= '<a href="javascript:;" data-tab="#wpex-mb-tab-' . $count . '">';

							if ( isset( $tab['icon'] ) ) {
								$tabs_output .= '<span class="' . esc_attr( $tab['icon'] ) .'" aria-hidden="true"></span>';
							}

							$tabs_output .= esc_html( $tab_title );

						$tabs_output .= '</a>';

					$tabs_output .= '</li>';

				}

			$tabs_output .= '</ul>';

			$metabox_html .= $tabs_output;

		}

		// Output tab sections.
		$count = 0;
		foreach ( $active_tabs as $tab_key => $tab ) {
			$count++;

			$is_open_tab = $has_active_tab = false;

			if ( ! $has_active_tab ) {
				if ( $open_tab ) {
					if ( $open_tab === $tab_key ) {
						$is_open_tab = true;
					}
				} elseif ( 1 == $count ) {
					$is_open_tab = true;
				}
			}

			$tab_class = 'wpex-mb-tab-panel';

			if ( $tab_count > 1 ) {
				$tab_class .= ' wp-tab-panel';
				if ( $is_open_tab && ! $has_active_tab ) {
					$tab_class .= ' wp-tab-panel-active';
					$has_active_tab = true;
				}
			}

			$metabox_html .= '<div id="wpex-mb-tab-' . absint( $count ) .'" class="' . esc_attr( $tab_class ) . '">';

				$metabox_html .= '<table class="form-table">';

					foreach ( $tab['settings'] as $setting ) {

						if ( isset( $setting['condition'] ) && ! $setting['condition'] ) {
							continue;
						}

						$meta_id     = $setting['id'];
						$type        = $setting['type'] ?? 'text';
						$description = $setting['description'] ?? '';
						$default     = $setting['default'] ?? '';
						$meta_value  = get_post_meta( $post_id, $meta_id, true );
						$meta_value  = $meta_value ?: $default;

						$tr_class    = '';

						if ( ! empty( $setting['hidden'] ) && true === $setting['hidden'] ) {
							$tr_class = ' class="wpex-mb-hidden"';
						}

						$metabox_html .= '<tr id="' . esc_attr( $meta_id ) . '_tr"' . $tr_class . '>';

							$metabox_html .= $this->get_setting_th(
								'wpex-mb-' . sanitize_html_class( str_replace( 'wpex_', '', $meta_id ) ),
								$setting['title'],
								$description
							);

							$type_method = 'get_field_' . $type;

							if ( method_exists( $this, $type_method ) ) {
								$metabox_html .= $this->$type_method( $setting, $meta_value );
							}

						$metabox_html .= '</tr>';

					}

				$metabox_html .= '</table>';

			$metabox_html .= '</div>';

		}

		if ( $this->has_core_fields() ) {
			$metabox_html .= $this->get_reset_buttons();
		}

		// @codingStandardsIgnoreLine
		echo $metabox_html;
	}

	/**
	 * Save metabox data.
	 */
	public function save_meta_data( $post_id ) {

		/*
		 * We need to verify this came from our screen and with proper authorization,
		 * because the save_post action can be triggered at other times.
		 */

		// Check if our nonce is set.
		if ( ! isset( $_POST['wpex_metabox_nonce'] ) ) {
			return;
		}

		// Verify that the nonce is valid.
		if ( ! wp_verify_nonce( $_POST['wpex_metabox_nonce'], 'wpex_metabox' ) ) {
			return;
		}

		// If this is an autosave, our form has not been submitted, so we don't want to do anything.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// Check the user's permissions.
		if ( isset( $_POST['post_type'] ) && 'page' === $_POST['post_type'] ) {

			if ( ! current_user_can( 'edit_page', $post_id ) ) {
				return;
			}

		} else {

			if ( ! current_user_can( 'edit_post', $post_id ) ) {
				return;
			}

		}

		/* OK, it's safe for us to save the data now. Now we can loop through fields */

		// Check reset field.
		$reset = $_POST['wpex_metabox_reset'] ?? '';

		// Get array of settings to save.
		$tabs = $this->meta_array( get_post( $post_id ) );

		// No tabs so lets bail.
		if ( ! $tabs ) {
			return;
		}

		// Get current post type.
		$post_type = get_post_type( $post_id );

		// Loop through tabs to locate the ones active on the post.
		$active_tabs = [];
		foreach ( $tabs as $tab_key => $tab ) {
			$tab_post_type = $this->get_tab_post_type( $tab );
			if ( ! $tab_post_type || in_array( $post_type, $tab_post_type ) ) {
				$active_tabs[$tab_key] = $tab;
			}
		}

		// Loop through tabs to grab all settings.
		$settings = [];
		foreach( $active_tabs as $tab ) {
			foreach ( $tab['settings'] as $setting ) {
				$settings[] = $setting;
			}
		}

		// No settings to check.
		if ( empty( $settings ) ) {
			return;
		}

		// Loop through settings and validate.
		foreach ( $settings as $setting ) {

			$id = $setting['id'];

			if ( 'on' === $reset ) {
				delete_post_meta( $post_id, $id );
				continue;
			}

			// Vars.
			$value = $_POST[ $id ] ?? '';
			$type  = $setting['type'] ?? 'text';

			switch ( $type ) {

				case 'checkbox':
					$value = $value ? 'on' : null;
					break;

				case 'text':
				case 'text_html':
				case 'code':
					if ( $value ) {
						$value = wp_kses_post( $value );
					}
					break;

				case 'date':
					if ( $value ) {
						$value = strtotime( wp_strip_all_tags( $value ) );
					}
					break;

				case 'iframe':
					if ( $value ) {
						$value = wp_kses( $value, array(
							'iframe' => array(
								'src'             => [],
								'height'          => [],
								'width'           => [],
								'frameborder'     => [],
								'allowfullscreen' => [],
								'allow'           => [],
							),
						) );
					}
					break;

				case 'textarea':
					if ( $value ) {
						$value = esc_html( $value );
					}
					break;

				case 'link':
					if ( $value ) {
						$value = esc_url( $value );
					}
					break;

				case 'select':

					if ( 'default' === $value ) {
						$value = ''; // the default value should save as empty.
					}

					if ( $value ) {
						$value = wp_strip_all_tags( $value ); // @todo compare to available options.
					}

					break;

				case 'media':
				case 'image':

					// Move old wpex_post_self_hosted_shortcode_redux to wpex_post_self_hosted_media.
					if ( 'wpex_post_self_hosted_media' === $id && empty( $value )
						&& $old = get_post_meta( $post_id, 'wpex_post_self_hosted_shortcode_redux', true )
					) {
						$value = $old;
						delete_post_meta( $post_id, 'wpex_post_self_hosted_shortcode_redux' );
					}

					if ( $value ) {
						$value = sanitize_text_field( $value );
					}
					break;

					case 'editor':

						$value = ( '<p><br data-mce-bogus="1"></p>' === $value ) ? '' : $value;

						if ( $value ) {
							$value = wp_kses_post( $value );
						}

					break;

				default:
					$value = sanitize_text_field( $value );
					break;

			} // end switch.

			// Update meta value.
			if ( $value ) {
				update_post_meta( $post_id, $id, $value );
			} else {
				delete_post_meta( $post_id, $id );
			}

		} // end foreach.

	}

	/**
	 * Settings Array.
	 */
	private function meta_array( $post = null ) {
		$array = [];

		if ( $this->has_core_fields() ) {
			$array['main']       = $this->get_main_fields();
			$array['header']     = $this->get_header_fields();
			$array['title']      = $this->get_title_fields();
			$array['slider']     = $this->get_slider_fields();
			$array['background'] = $this->get_background_fields();
			$array['footer']     = $this->get_footer_fields();
			$array['callout']    = $this->get_callout_fields();
		}

		$array['media'] = $this->get_media_fields();

		/**
		 * Filters the metabox settings.
		 *
		 * @param array $settings
		 * @param object $post
		 */
		$array = (array) apply_filters( 'wpex_metabox_array', $array, $post );

		return $array;
	}

	/**
	 * Returns the main tab fields.
	 */
	protected function get_main_fields() {
		return array(
			'title' => esc_html__( 'Main', 'total-theme-core' ),
			'settings' => array(
				'post_link' => array(
					'title' => esc_html__( 'Redirect', 'total-theme-core' ),
					'id' => 'wpex_post_link',
					'type' => 'link',
					'description' => esc_html__( 'Enter a URL to redirect this post or page.', 'total-theme-core' ),
				),
				'main_layout' =>array(
					'title' => esc_html__( 'Site Layout', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_main_layout',
					'description' => esc_html__( 'This option should only be used in very specific cases since there is a global setting available in the Customizer.', 'total-theme-core' ),
					'options_callback' => 'wpex_get_site_layouts',
				),
				'post_layout' => array(
					'title' => esc_html__( 'Content Layout', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_post_layout',
					'description' => esc_html__( 'Select your custom layout for this page or post content.', 'total-theme-core' ),
					'options_callback' => 'wpex_get_post_layouts',
				),
				'singular_template'    => array(
					'title' => esc_html__( 'Dynamic Template', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_singular_template',
					'description' => esc_html__( 'Select a dynamic templatera template to override this page. If selected it will disable the front-end editor.', 'total-theme-core' ),
					'options_callback' => array( $this, 'get_template_choices' ),
				),
				'sidebar' => array(
					'title' => esc_html__( 'Sidebar', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'sidebar',
					'description' => esc_html__( 'Select your a custom sidebar for this page or post.', 'total-theme-core' ),
					'options_callback' => array( $this, 'get_widget_areas' ),
				),
				'disable_toggle_bar'   => array(
					'title' => esc_html__( 'Toggle Bar', 'total-theme-core' ),
					'id' => 'wpex_disable_toggle_bar',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options'          => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'enable' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'disable_top_bar' => array(
					'title' => esc_html__( 'Top Bar', 'total-theme-core' ),
					'id' => 'wpex_disable_top_bar',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'enable' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'disable_breadcrumbs' => array(
					'title' => esc_html__( 'Breadcrumbs', 'total-theme-core' ),
					'id' => 'wpex_disable_breadcrumbs',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'enable' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'disable_social' => array(
					'title' => esc_html__( 'Social Share', 'total-theme-core' ),
					'id' => 'wpex_disable_social',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'enable' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'secondary_thumbnail' => array(
					'title' => esc_html__( 'Secondary Image', 'total-theme-core' ),
					'id' => 'wpex_secondary_thumbnail',
					'type' => 'image',
					'description' => esc_html__( 'Used for the secondary Image Swap overlay style.', 'total-theme-core' ),
				),
			),
		);
	}

	/**
	 * Returns the header tab fields.
	 */
	protected function get_header_fields() {
		$is_dev_header = ( 'dev' === get_theme_mod( 'header_style' ) );

		return array(
			'title' => esc_html__( 'Header', 'total-theme-core' ),
			'settings' => array(
				'disable_header' => array(
					'title' => esc_html__( 'Header', 'total-theme-core' ),
					'id' => 'wpex_disable_header',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'enable' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'header_style' => array(
					'title' => esc_html__( 'Header Style', 'total-theme-core' ),
					'id' => 'wpex_header_style',
					'type' => 'select',
					'description' => esc_html__( 'Override default header style.', 'total-theme-core' ),
					'options' => $this->get_header_styles(),
					'condition' => ! $is_dev_header,
				),
				'sticky_header' => array(
					'title' => esc_html__( 'Sticky Header', 'total-theme-core' ),
					'id' => 'wpex_sticky_header',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'enable' => esc_html__( 'Enable', 'total-theme-core' ),
						'disable' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'logo_scroll_top' => array(
					'title' => esc_html__( 'Scroll Up When Clicking Logo', 'total-theme-core' ),
					'id' => 'wpex_logo_scroll_top',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'enable' => esc_html__( 'Enable', 'total-theme-core' ),
						'disable' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'header_menu' => array(
					'title' => esc_html__( 'Menu', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_header_menu',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'on' => esc_html__( 'Enable', 'total-theme-core' ),
						'off' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'custom_menu' => array(
					'title' => esc_html__( 'Custom Menu', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_custom_menu',
					'description' => esc_html__( 'Select a custom menu for this page or post.', 'total-theme-core' ),
					'options' => $this->get_menus(),
				),
				'overlay_header' => array(
					'title' => esc_html__( 'Transparent Header', 'total-theme-core' ),
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'id' => 'wpex_overlay_header',
					'type' => 'select',
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'on' => esc_html__( 'Enable', 'total-theme-core' ),
						'off' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'overlay_header_style' => array(
					'title' => esc_html__( 'Transparent Header Style', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_overlay_header_style',
					'description' => esc_html__( 'Select your overlay header style', 'total-theme-core' ),
					'options_callback' => 'wpex_header_overlay_styles',
					'default' => '',
					'condition' => ! $is_dev_header,
				),
				'overlay_header_background' => array(
					'title' => esc_html__( 'Transparent Header Background', 'total-theme-core' ),
					'id' => 'wpex_overlay_header_background',
					'description' => esc_html__( 'Select a color to enable a background for your header (optional)', 'total-theme-core' ),
					'type' => 'color',
				),
				'overlay_header_dropdown_style' => array(
					'title' => esc_html__( 'Transparent Header Dropdown Style', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_overlay_header_dropdown_style',
					'description' => esc_html__( 'Select your overlay header style', 'total-theme-core' ),
					'options_callback' => 'wpex_get_menu_dropdown_styles',
					//'default' => 'black', // @deprecated 1.0.4
					'condition' => ! $is_dev_header,
				),
				'overlay_header_font_size' => array(
					'title' => esc_html__( 'Transparent Header Menu Font Size', 'total-theme-core'),
					'id' => 'wpex_overlay_header_font_size',
					'description' => esc_html__( 'Enter a size in px.', 'total-theme-core' ),
					'type' => 'number',
					'max' => '99',
					'min' => '8',
					'condition' => ! $is_dev_header,
				),
				'overlay_header_logo' => array(
					'title' => esc_html__( 'Transparent Header Logo', 'total-theme-core'),
					'id' => 'wpex_overlay_header_logo',
					'type' => 'image',
					'description' => esc_html__( 'Select a custom logo (optional) for the overlay header.', 'total-theme-core' ),
				),
				'overlay_header_logo_retina' => array(
					'title' => esc_html__( 'Transparent Header Logo: Retina', 'total-theme-core'),
					'id' => 'wpex_overlay_header_logo_retina',
					'type' => 'image',
					'description' => esc_html__( 'Retina version for the overlay header custom logo.', 'total-theme-core' ),
				),
			),
		);
	}

	/**
	 * Returns the title tab fields.
	 */
	protected function get_title_fields() {
		return array(
			'title' => esc_html__( 'Title', 'total-theme-core' ),
			'settings' => array(
				'disable_title' => array(
					'title' => esc_html__( 'Title', 'total-theme-core' ),
					'id' => 'wpex_disable_title',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'enable' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'post_title' => array(
					'title' => esc_html__( 'Custom Title', 'total-theme-core' ),
					'id' => 'wpex_post_title',
					'type' => 'text',
					'description' => esc_html__( 'Alter the main title display.', 'total-theme-core' ),
				),
				'disable_header_margin' => array(
					'title' => esc_html__( 'Title Margin', 'total-theme-core' ),
					'id' => 'wpex_disable_header_margin',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'post_subheading' => array(
					'title' => esc_html__( 'Subheading', 'total-theme-core' ),
					'type' => 'text_html',
					'id' => 'wpex_post_subheading',
					'description' => esc_html__( 'Enter your page subheading. Shortcodes & HTML is allowed.', 'total-theme-core' ),
				),
				'post_title_style' => array(
					'title' => esc_html__( 'Title Style', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_post_title_style',
					'description' => esc_html__( 'Select a custom title style for this page or post.', 'total-theme-core' ),
					'options' => $this->get_title_styles(),
				),
				'post_title_background_color' => array(
					'title' => esc_html__( 'Background Color', 'total-theme-core' ),
					'description' => esc_html__( 'Select a color.', 'total-theme-core' ),
					'id' => 'wpex_post_title_background_color',
					'type' => 'color',
					'hidden' => true,
				),
				'post_title_background_redux' => array(
					'title' => esc_html__( 'Background Image', 'total-theme-core'),
					'id' => 'wpex_post_title_background_redux', //@todo remove _redux
					'type' => 'image',
					'description' => esc_html__( 'Select a custom header image for your main title.', 'total-theme-core' ),
					'hidden' => true,
				),
				'post_title_height' => array(
					'title' => esc_html__( 'Background Height', 'total-theme-core' ),
					'type' => 'text',
					'id' => 'wpex_post_title_height',
					'description' => esc_html__( 'Select your custom height for your title background.', 'total-theme-core' ),
					'hidden' => true,
				),
				'post_title_background_style' => array(
					'title' => esc_html__( 'Background Style', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_post_title_background_image_style',
					'description' => esc_html__( 'Select the style.', 'total-theme-core' ),
					'options_callback' => 'wpex_get_bg_img_styles',
					'hidden' => true,
				),
				'post_title_background_position' => array(
					'title' => esc_html__( 'Background Position', 'total-theme-core' ),
					'type' => 'text',
					'id' => 'wpex_post_title_background_position', // @todo rename to post_title_background_image_position
					'description' => esc_html__( 'Enter a custom position for your background image.', 'total-theme-core' ),
					'hidden' => true,
				),
				'post_title_background_overlay' => array(
					'title' => esc_html__( 'Background Overlay', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_post_title_background_overlay',
					'description' => esc_html__( 'Select an overlay for the title background.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'None', 'total-theme-core' ),
						'dark' => esc_html__( 'Dark', 'total-theme-core' ),
						'dotted' => esc_html__( 'Dotted', 'total-theme-core' ),
						'dashed' => esc_html__( 'Diagonal Lines', 'total-theme-core' ),
						'bg_color' => esc_html__( 'Background Color', 'total-theme-core' ),
					),
					'hidden' => true,
				),
				'post_title_background_overlay_opacity' => array(
					'id' => 'wpex_post_title_background_overlay_opacity',
					'type' => 'number',
					'title' => esc_html__( 'Background Overlay Opacity', 'total-theme-core' ),
					'description' => esc_html__( 'Enter a custom opacity for your title background overlay.', 'total-theme-core' ),
					'default' => '',
					'hidden' => true,
					'step' => 0.01,
					'min' => 0,
					'max' => 1,
				),
			),
		);
	}

	/**
	 * Returns the backgrounds tab fields.
	 */
	protected function get_background_fields() {
		return array(
			'title' => esc_html__( 'Background', 'total-theme-core' ),
			'settings' => array(
				'page_background_color' => array(
					'title' => esc_html__( 'Background Color', 'total-theme-core' ),
					'description' => esc_html__( 'Select a color.', 'total-theme-core' ),
					'id' => 'wpex_page_background_color',
					'type' => 'color',
				),

				// @todo remove _redux
				'page_background_image_redux' => array(
					'title' => esc_html__( 'Background Image', 'total-theme-core' ),
					'id' => 'wpex_page_background_image_redux',
					'description' => esc_html__( 'Select an image.', 'total-theme-core' ),
					'type' => 'image',
				),
				'page_background_image_style' => array(
					'title' => esc_html__( 'Background Style', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_page_background_image_style',
					'description' => esc_html__( 'Select the style.', 'total-theme-core' ),
					'options_callback' => 'wpex_get_bg_img_styles',
				),
			),
		);
	}

	/**
	 * Returns the footer tab fields.
	 */
	protected function get_footer_fields() {
		return array(
			'title' => esc_html__( 'Footer', 'total-theme-core' ),
			'settings' => array(
				'disable_footer' => array(
					'title' => esc_html__( 'Footer', 'total-theme-core' ),
					'id' => 'wpex_disable_footer',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'enable' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'disable_footer_widgets' => array(
					'title' => esc_html__( 'Footer Widgets', 'total-theme-core' ),
					'id' => 'wpex_disable_footer_widgets',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'footer_reveal' => array(
					'title' => esc_html__( 'Footer Reveal', 'total-theme-core' ),
					'description' => esc_html__( 'Enable the footer reveal style. The footer will be placed in a fixed postion and display on scroll. This setting is for the "Full-Width" layout only and desktops only.', 'total-theme-core' ),
					'id' => 'wpex_footer_reveal',
					'type' => 'select',
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'on' => esc_html__( 'Enable', 'total-theme-core' ),
						'off' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'footer_bottom' => array(
					'title' => esc_html__( 'Footer Bottom', 'total-theme-core' ),
					'description' => esc_html__( 'Enable the footer bottom area (copyright section).', 'total-theme-core' ),
					'id' => 'wpex_footer_bottom',
					'type' => 'select',
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'on' => esc_html__( 'Enable', 'total-theme-core' ),
						'off' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
			),
		);
	}

	/**
	 * Returns the callout tab fields.
	 */
	protected function get_callout_fields() {
		return array(
			'title' => esc_html__( 'Callout', 'total-theme-core' ),
			//'icon' => 'dashicons dashicons-megaphone',
			'settings' => array(
				'disable_footer_callout' => array(
					'title' => esc_html__( 'Callout', 'total-theme-core' ),
					'id' => 'wpex_disable_footer_callout',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable this element on this page or post.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'enable' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'callout_link' => array(
					'title' => esc_html__( 'Callout Link', 'total-theme-core' ),
					'id' => 'wpex_callout_link',
					'type' => 'link',
					'description' => esc_html__( 'Enter a valid link.', 'total-theme-core' ),
				),
				'callout_link_txt' => array(
					'title' => esc_html__( 'Callout Link Text', 'total-theme-core' ),
					'id' => 'wpex_callout_link_txt',
					'type' => 'text',
					'description' => esc_html__( 'Enter your text.', 'total-theme-core' ),
				),
				'callout_text' => array(
					'title' => esc_html__( 'Callout Text', 'total-theme-core' ),
					'id' => 'wpex_callout_text',
					'type' => 'editor',
					'rows' => '5',
					'teeny' => true,
					'media_buttons' => false,
					'description' => esc_html__( 'Override the default callout text and if your callout box is disabled globally but you have content here it will still display for this page or post.', 'total-theme-core' ),
				),
			),
		);
	}

	/**
	 * Returns the slider tab fields.
	 */
	protected function get_slider_fields() {
		return array(
			'title' => esc_html__( 'Slider', 'total-theme-core' ),
			'settings' => array(
				'post_slider_shortcode' => array(
					'title' => esc_html__( 'Slider Shortcode', 'total-theme-core' ),
					'type' => 'code',
					'rows' => 2,
					'id' => 'wpex_post_slider_shortcode',
					'description' => esc_html__( 'Enter a slider shortcode here to display a slider at the top of the page.', 'total-theme-core' ),
				),
				'post_slider_shortcode_position' => array(
					'title' => esc_html__( 'Slider Position', 'total-theme-core' ),
					'type' => 'select',
					'id' => 'wpex_post_slider_shortcode_position',
					'description' => esc_html__( 'Select the position for the slider shortcode.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'below_title' => esc_html__( 'Below Title', 'total-theme-core' ),
						'above_title' => esc_html__( 'Above Title', 'total-theme-core' ),
						'above_menu' => esc_html__( 'Above Menu (Header 2 or 3)', 'total-theme-core' ),
						'above_header' => esc_html__( 'Above Header', 'total-theme-core' ),
						'above_topbar' => esc_html__( 'Above Top Bar', 'total-theme-core' ),
					),
				),
				'post_slider_bottom_margin' => array(
					'title' => esc_html__( 'Slider Bottom Margin', 'total-theme-core' ),
					'description' => esc_html__( 'Enter a bottom margin for your slider in pixels.', 'total-theme-core' ),
					'id' => 'wpex_post_slider_bottom_margin',
					'type' => 'text',
				),
				'contain_post_slider' => array(
					'title' => esc_html__( 'Contain Slider?', 'total-theme-core' ),
					'id' => 'wpex_contain_post_slider',
					'type' => 'select',
					'description' => esc_html__( 'Adds the container wrapper around the slider to center it with the rest of the content.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Disable', 'total-theme-core' ),
						'on' => esc_html__( 'Enable', 'total-theme-core' ),
					),
				),
				'disable_post_slider_mobile' => array(
					'title' => esc_html__( 'Slider On Mobile', 'total-theme-core' ),
					'id' => 'wpex_disable_post_slider_mobile',
					'type' => 'select',
					'description' => esc_html__( 'Enable or disable slider display for mobile devices.', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Enable', 'total-theme-core' ),
						'on' => esc_html__( 'Disable', 'total-theme-core' ),
					),
				),
				'post_slider_mobile_alt' => array(
					'title' => esc_html__( 'Slider Mobile Alternative', 'total-theme-core' ),
					'type' => 'media',
					'id' => 'wpex_post_slider_mobile_alt',
					'description' => esc_html__( 'Select an image.', 'total-theme-core' ),
					'type' => 'image',
				),
				'post_slider_mobile_alt_url' => array(
					'title' => esc_html__( 'Slider Mobile Alternative URL', 'total-theme-core' ),
					'id' => 'wpex_post_slider_mobile_alt_url',
					'description' => esc_html__( 'URL for the mobile slider alternative.', 'total-theme-core' ),
					'type' => 'text',
				),
				'post_slider_mobile_alt_url_target' => array(
					'title' => esc_html__( 'Slider Mobile Alternative URL Target', 'total-theme-core' ),
					'id' => 'wpex_post_slider_mobile_alt_url_target',
					'description' => esc_html__( 'Select your link target window.', 'total-theme-core' ),
					'type' => 'select',
					'options' => array(
						'' => esc_html__( 'Same Window', 'total-theme-core' ),
						'blank' => esc_html__( 'New Window', 'total-theme-core' ),
					),
				),
			),
		);
	}

	/**
	 * Returns the slider tab fields.
	 */
	protected function get_media_fields() {
		return array(
			'title' => esc_html__( 'Media', 'total-theme-core' ),
			'post_type' => array( 'post' ),
			'settings' => array(
				'post_media_position' => array(
					'title' => esc_html__( 'Media Display/Position', 'total-theme-core' ),
					'id' => 'wpex_post_media_position',
					'type' => 'select',
					'description' => esc_html__( 'Select your preferred position for your post\'s media (featured image or video).', 'total-theme-core' ),
					'options' => array(
						'' => esc_html__( 'Default', 'total-theme-core' ),
						'above' => esc_html__( 'Full-Width Above Content', 'total-theme-core' ),
						'hidden' => esc_html__( 'None (Do Not Display Featured Image/Video)', 'total-theme-core' ),
					),
				),
				'post_oembed' => array(
					'title' => esc_html__( 'oEmbed URL', 'total-theme-core' ),
					'description' => esc_html__( 'Enter a URL that is compatible with WP\'s built-in oEmbed feature. This setting is used for your video and audio post formats.', 'total-theme-core' ) . '<br><a href="http://codex.wordpress.org/Embeds" target="_blank">'. esc_html__( 'Learn More', 'total-theme-core' ) . ' &rarr;</a>',
					'id' => 'wpex_post_oembed',
					'type' => 'text',
				),
				'post_self_hosted_shortcode_redux' => array(
					'title' => esc_html__( 'Self Hosted', 'total-theme-core' ),
					'description' => esc_html__( 'Insert your self hosted video or audio URL here.', 'total-theme-core' ) . '<br><a href="http://make.wordpress.org/core/2013/04/08/audio-video-support-in-core/" target="_blank">' . esc_html__( 'Learn More', 'total-theme-core' ) . ' &rarr;</a>',
					'id' => 'wpex_post_self_hosted_media',
					'type' => 'media',
				),
				'post_video_embed' => array(
					'title' => esc_html__( 'Embed Code', 'total-theme-core' ),
					'description' => esc_html__( 'Insert your embed/iframe code.', 'total-theme-core' ),
					'id' => 'wpex_post_video_embed',
					'type' => 'iframe',
					'rows' => 4,
				),
			),
		);
	}

	/**
	 * Get toolset types.
	 */
	protected function add_toolset_types() {
		$types = (array) get_option( 'wpcf-custom-types' );
		if ( $types ) {
			$public_types = [];
			foreach( $types as $type => $params ) {
				if ( ! empty( $params['public'] ) ) {
					$this->post_types[$type] = $type;
				}
			}
		}
	}

	/**
	 * Get tab screen (post_type).
	 */
	protected function get_tab_post_type( $tab ) {
		$type = $tab['post_type'] ?? [];
		if ( is_string( $type ) ) {
			return str_split( $type );
		}
		return $type;
	}

	/**
	 * Checks if the core fields are enabled.
	 */
	protected function has_core_fields() {
		return get_theme_mod( 'theme_settings_metabox_core_fields_enable', true );
	}

	/**
	 * Checks if we are currently editing the footer builder page.
	 */
	protected function is_footer_builder_page( $post ) {
		$footer_builder_page = get_theme_mod( 'footer_builder_page_id' );
		if ( $footer_builder_page && 'page' === get_post_type( $post->ID ) && $footer_builder_page == $post->ID ) {
			return true;
		}
	}

	/**
	 * Get header_styles.
	 */
	protected function get_header_styles() {
		$header_styles = array(
			'' => esc_html__( 'Default', 'total-theme-core' ),
		);

		if ( function_exists( 'wpex_get_header_styles' ) ) {
			$header_styles = $header_styles + wpex_get_header_styles();
		}
		return $header_styles;
	}

	/**
	 * Get menus.
	 */
	protected function get_menus() {
		$menus = array( esc_html__( 'Default', 'total-theme-core' ) );
		$get_menus = get_terms( 'nav_menu', array( 'hide_empty' => true ) );
		foreach ( $get_menus as $menu) {
			$menus[$menu->term_id] = $menu->name;
		}
		return $menus;
	}

	/**
	 * Get title styles.
	 */
	protected function get_title_styles() {
		return apply_filters( 'wpex_title_styles', array(
			''                 => esc_html__( 'Default', 'total-theme-core' ),
			'centered'         => esc_html__( 'Centered', 'total-theme-core' ),
			'centered-minimal' => esc_html__( 'Centered Minimal', 'total-theme-core' ),
			'background-image' => esc_html__( 'Background Image', 'total-theme-core' ),
			'solid-color'      => esc_html__( 'Solid Color & White Text', 'total-theme-core' ),
		) );
	}

	/**
	 * Get widget areas.
	 */
	protected function get_widget_areas() {
		$widget_areas = array( esc_html__( 'Default', 'total-theme-core' ) );
		if ( function_exists( 'wpex_get_widget_areas' ) ) {
			$widget_areas = $widget_areas + wpex_get_widget_areas();
		}
		return $widget_areas;
	}

	/**
	 * Get templatera templates.
	 */
	protected function get_template_choices() {
		if ( class_exists( 'TotalTheme\\Theme_Builder', false )
			&& is_callable( [ 'TotalTheme\\Theme_Builder', 'get_template_post_types' ] )
		) {
			$types = Theme_Builder::get_template_post_types();
		} else {
			$types = [ 'templatera' ];
		}
		$choices = [ esc_html__( 'Default', 'total-theme-core' ) ];
		$get_templates = new WP_Query( [
			'posts_per_page' => -1,
			'post_type'      => array_keys( $types ),
			'fields'         => 'ids',
		] );
		$multi_array = count( $types ) > 1;
		if ( $get_templates->posts ) {
			foreach ( $get_templates->posts as $template ) {
				$template = absint( $template );
				if ( $multi_array ) {
					$type = get_post_type( $template );
					if ( ! isset( $choices[$type] ) ) {
						$choices[$type] = [
							'label'   => $types[$type] ?? $type,
							'choices' => [],
						];
					}
					$choices[$type]['choices'][$template] = wp_strip_all_tags( get_the_title( $template ) );
				} else {
					$choices[$template] = wp_strip_all_tags( get_the_title( $template ) );
				}
			}
		}
		return $choices;
	}

	/**
	 * Returns setting th.
	 */
	protected function get_setting_th( $field_id, $title, $description ) {
		$html = '<th>';
			$html .= '<label class="wpex-metabox__label" for="' . esc_attr( $field_id ) . '">';
				$allowed_tags = array(
					'br' => true,
					'span' => array(
						'class' => true,
						'aria-hidden' => true,
					),
					'svg' => array(
						'class' => true,
						'aria-hidden' => true,
						'aria-labelledby' => true,
						'role' => true,
						'xmlns' => true,
						'width' => true,
						'height' => true,
						'viewbox' => true, // <= Must be lower case!
						'style' => true,
						'preserveAspectRatio' => true, // WP bug prevents this from working :(
					),
					'g' => array( 'fill' => true ),
					'title' => array( 'title' => true ),
					'path' => array(
						'd' => true,
						'fill' => true,
						'class' => true,
					),
				);
				$html .= wp_kses( $title, $allowed_tags );
			$html .= '</label>';
			if ( ! empty( $description ) ) {
				$html .= '<p class="wpex-mb-description">' . wp_kses_post( $description ) . '</p>';
			}
		$html .= '</th>';
		return $html;
	}

	/**
	 * Returns field name.
	 */
	protected function get_field_name( $field ) {
		return $field['id'];
	}

	/**
	 * Returns field id.
	 */
	protected function get_field_id( $field ) {
		return 'wpex-mb-' . sanitize_html_class( str_replace( 'wpex_', '', $field['id'] ) );
	}

	/**
	 * Returns text field.
	 */
	protected function get_field_text( $field, $value ) {
		return '<td><input id="' . esc_attr( $this->get_field_id( $field ) ) . '" class="wpex-input" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="text" value="' . esc_attr( $value ) . '"></td>';
	}

	/**
	 * Returns button group field.
	 */
	protected function get_field_button_group( $field, $value ) {
		$options = $field['options'] ?? '';

		if ( ! is_array( $options ) ) {
			return;
		}

		$html = '<td>';

			$html .= '<div class="wpex-mb-btn-group">';

				foreach ( $options as $option_value => $option_name ) {

					$class = 'wpex-mb-btn wpex-mb-' . esc_attr( $option_value );

					if ( $option_value == $value ) {
						$class .= ' active';
					}

					$html .= '<button type="button" class="' . esc_attr( $class ) . '" data-value="' . esc_attr( $option_value ) . '">' . esc_html( $option_name ) . '</button>';

				}

				$html .= '<input id="' . esc_attr( $this->get_field_id( $field ) ) . '" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="hidden" value="' . esc_attr( $value ) . '" class="wpex-mb-hidden">';

			$html .= '</div>';

		$html .= '</td>';

		return $html;
	}

	/**
	 * On/off switch.
	 */
	protected function get_field_button_group_ed( $field, $value ) {
		$options = $field['options'] ?? '';

		if ( ! is_array( $options ) ) {
			return;
		}

		$html = '<td>';

			$html .= '<div class="wpex-mb-btn-group">';

				// Default.
				$active = ! $field ? 'wpex-mb-btn wpex-default active' : 'wpex-mb-btn wpex-default';

				$html .= '<button type="button" class="' . esc_attr( $active ) . '" data-value="">' . esc_html__( 'Default', 'total-theme-core' ) . '</button>';

				// Enable.
				$active = ( $options['enable'] == $field ) ? 'wpex-mb-btn wpex-on active' : 'wpex-mb-btn wpex-on';

				$html .= '<button type="button" class="' . esc_attr( $active ) . '" data-value="' . esc_attr( $options['enable'] ) . '">' . esc_html__( 'Enable', 'total-theme-core' ) . '</button>';

				// Disable.
				$active = ( $options['disable'] == $field ) ? 'wpex-mb-btn wpex-off active' : 'wpex-mb-btn wpex-off';

				$html .= '<button type="button" class="' . esc_attr( $active ) . '" data-value="' . esc_attr( $options['disable'] ) . '">' . esc_html__( 'Disable', 'total-theme-core' )  . '</button>';

				$html .= '<input id="' . esc_attr( $this->get_field_id( $field ) ) . '" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="hidden" value="' . esc_attr( $value ) . '" class="wpex-mb-hidden">';

			$html .= '</div>';

		$html .= '</td>';

		return $html;
	}

	/**
	 * Date field.
	 */
	protected function get_field_date( $field, $value ) {
		$value = ! empty( $value ) ? date( get_option( 'date_format' ), $value ) : '';

		wp_enqueue_script( 'jquery-ui' );
		wp_enqueue_script( 'jquery-ui-datepicker', array( 'jquery-ui' ) );

		if ( function_exists( 'wp_localize_jquery_ui_datepicker' ) ) {
			wp_localize_jquery_ui_datepicker();
		}

		wp_enqueue_style(
			'jquery-ui-datepicker-style',
			'//ajax.googleapis.com/ajax/libs/jqueryui/1.10.4/themes/smoothness/jquery-ui.css'
		);

		return '<td><input id="' . esc_attr( $this->get_field_id( $field ) ) . '" class="wpex-input wpex-date-meta" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="text" value="' . esc_attr( $value ) . '"></td>';
	}

	/**
	 * Number field.
	 */
	protected function get_field_number( $field, $value ) {
		$step = $field['step'] ?? '1';
		$min  = $field['min'] ?? '1';
		$max  = $field['max'] ?? '10';

		$html = '<td>';
			$html .= '<input id="' . esc_attr( $this->get_field_id( $field ) ) . '" class="wpex-input" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="number" value="' . esc_attr( $value ) . '" step="' . floatval( $step ) . '" min="' . floatval( $min ) . '" max="' . floatval( $max ) . '">';
		$html .= '</td>';

		return $html;
	}

	/**
	 * Number field.
	 */
	protected function get_field_text_html( $field, $value ) {
		return '<td><input id="' . esc_attr( $this->get_field_id( $field ) ) . '" class="wpex-input" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="text" value="' . esc_attr( $value ) . '"></td>';
	}

	/**
	 * Link field.
	 */
	protected function get_field_link( $field, $value ) {
		$value = ( 'home_url' === $value ) ? esc_html( $value ) : esc_url( $value );

		return '<td><input id="' . esc_attr( $this->get_field_id( $field ) ) . '" class="wpex-input" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="text" value="' . esc_attr( $value ) . '"></td>';
	}

	/**
	 * Textarea field.
	 */
	protected function get_field_textarea( $field, $value ) {
		$rows = isset ( $field['rows'] ) ? absint( $field['rows'] ) : 4;

		$html = '<td>';
			$html .= '<textarea id="' . esc_attr( $this->get_field_id( $field ) ) . '" rows="' . esc_attr( $rows ) . '" cols="1" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="text" class="wpex-mb-textarea">' . esc_textarea( $value ) . '</textarea>';
		$html .= '</td>';

		return $html;
	}

	/**
	 * Code field.
	 */
	protected function get_field_code( $field, $value ) {
		$rows = isset ( $field['rows'] ) ? absint( $field['rows'] ) : 1;

		$html = '<td>';
			$html .= '<pre><textarea id="' . esc_attr( $this->get_field_id( $field ) ) . '" rows="' . esc_attr( $rows ) . '" cols="1" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="text" class="wpex-mb-textarea-code">' . esc_html( $value ) . '</textarea></pre>';
		$html .= '</td>';
		return $html;
	}

	/**
	 * iFrame field.
	 */
	protected function get_field_iframe( $field, $value ) {
		$rows = isset ( $field['rows'] ) ? absint( $field['rows'] ) : 1;

		$value_safe = wp_kses( $value, [
			'iframe' => [
				'src'             => [],
				'height'          => [],
				'width'           => [],
				'frameborder'     => [],
				'allowfullscreen' => [],
				'allow'           => [],
			],
		] );

		$html = '<td>';
			$html .= '<pre><textarea id="' . esc_attr( $this->get_field_id( $field ) ) . '" rows="' . esc_attr( $rows ) . '" cols="1" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="text" class="wpex-mb-textarea-code">' . $value_safe . '</textarea></pre>';
		$html .= '</td>';

		return $html;
	}

	/**
	 * Checkbox field.
	 */
	protected function get_field_checkbox( $field, $value ) {
		$value = ( 'on' !== $value ) ? false : true;

		return '<td><input name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="checkbox" ' . checked( $value, true, false ) . '></td>';
	}

	/**
	 * Select field.
	 */
	protected function get_field_select( $field, $value ) {
		if ( isset( $field['options_callback'] ) && is_callable( $field['options_callback'] ) ) {
			$field['options'] = call_user_func( $field['options_callback'] );
		}

		$options = $field['options'] ?? '';

		$html = '';

		if ( ! empty( $options ) && is_array( $options ) ) {
			$html .= '<td><select id="' . esc_attr( $this->get_field_id( $field ) ) . '" name="' . esc_attr( $this->get_field_name( $field ) ) . '">';
			foreach ( $options as $option_value => $option_name ) {
				if ( is_array( $option_name ) && isset( $option_name['choices'] ) ) {
					$html .= '<optgroup label="' . esc_attr( $option_name['label'] ) . '">';
					foreach ( $option_name['choices'] as $sub_option => $sub_option_label ) {
						$html .= '<option value="' . esc_attr( $sub_option ) . '" ' . selected( $value, $sub_option, false ) . '>' . esc_attr( $sub_option_label ) . '</option>';
					}
					$html .= '</optgroup>';
				} else {
					$html .= '<option value="' . esc_attr( $option_value ) . '" ' . selected( $value, $option_value, false ) . '>' . esc_attr( $option_name ) . '</option>';
				}
			}
			$html .= '</select></td>';
		} else {
			$html .= '<td><input id="' . esc_attr( $this->get_field_id( $field ) ) . '" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="text" value="' . esc_attr( $value ) . '"></td>';
		}

		return $html;
	}

	/**
	 * Color field.
	 */
	protected function get_field_color( $field, $value ) {
		return '<td><input id="' . esc_attr( $this->get_field_id( $field ) ) . '" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="text" value="' . esc_attr( $value ) . '" class="wpex-mb-color-field" data-alpha-enabled="true"></td>';
	}

	/**
	 * Image field.
	 */
	protected function get_field_image( $field, $value ) {
		// Validate data if array - old Redux cleanup
		if ( is_array( $value ) ) {
			if ( ! empty( $value['url'] ) ) {
				$value = $value['url'];
			} else {
				$value = '';
			}
		}

		$html = '<td>';
			$html .= '<div class="wpex-image-select">';
				$html .= '<input id="' . esc_attr( $this->get_field_id( $field ) ) . '" class="wpex-input" type="text" name="' . esc_attr( $this->get_field_name( $field ) ) . '" value="' . esc_attr( $value ) . '">';
				$html .= '<button class="wpex-mb-uploader button-primary" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="button">' . esc_html__( 'Select', 'total-theme-core' ) . '</button>';
			$html .= '</div>';
			$html .= '<div class="wpex-img-holder">';
				if ( $value ) {
					if ( is_numeric( $value ) && wp_attachment_is_image( $value ) ) {
						$html .= wp_get_attachment_image( $value, 'thumbnail' );
					} else {
						$html .= '<img src="' . esc_url( $value ) . '">';
					}
				}
			$html .= '</div>';
		$html .= '</td>';

		return $html;
	}

	/**
	 * Media field.
	 */
	protected function get_field_media( $field, $value ) {
		// Validate data if array - old Redux cleanup
		if ( is_array( $value ) ) {
			if ( ! empty( $value['url'] ) ) {
				$value = $value['url'];
			} else {
				$value = '';
			}
		}

		$html = '<td>';
			$html .= '<div class="uploader">';
			$html .= '<input id="' . esc_attr( $this->get_field_id( $field ) ) . '" class="wpex-input" type="text" name="' . esc_attr( $this->get_field_name( $field ) ) . '" value="' . esc_attr( $value ) . '">';
			$html .= '<input class="wpex-mb-uploader button-secondary" name="' . esc_attr( $this->get_field_name( $field ) ) . '" type="button" value="' . esc_html__( 'Upload', 'total-theme-core' ) . '">';
			$html .= '</div>';
		$html .= '</td>';

		return $html;
	}

	/**
	 * Editor field.
	 */
	protected function get_field_editor( $field, $value ) {
		$args = [
			'editor_id'     => esc_attr( $this->get_field_id( $field ) ) ,
			'textarea_name' => $this->get_field_name( $field ),
			'teeny'         => $field['teeny'] ?? false,
			'textarea_rows' => $field['rows'] ?? '10',
			'media_buttons' => $field['media_buttons'] ?? true,
			'echo'          => false,
		];

		ob_start();
			wp_editor( $value, esc_attr( $this->get_field_name( $field ) ), $args );
		$editor = ob_get_clean();

		return '<td>' . $editor . '</td>';
	}

	/**
	 * Returns the reset buttons.
	 */
	protected function get_reset_buttons() {
		$html = '<div class="wpex-metabox__footer">';
			$html .= '<div class="wpex-metabox__reset wpex-mb-reset">';
				$html .= '<button class="button button-secondary wpex-reset-btn" aria-controls="#wpex-metabox-reset-checkbox" aria-pressed="false">' . esc_html__( 'Reset Settings', 'total-theme-core' ) . '</button>';
				$html .= '<div id="wpex-metabox-reset-checkbox" class="wpex-metabox__reset-checkbox wpex-reset-checkbox"><input type="checkbox" name="wpex_metabox_reset" id="wpex-metabox-reset-checkbox-input"> <label for="wpex-metabox-reset-checkbox-input">' . esc_html__( 'Are you sure? Check this box, then update your post to reset all settings.', 'total-theme-core' ) . '</label></div>';
			$html .= '</div>';
		$html .= '</div>';
		return $html;
	}

}